<?php

namespace Parking;

use Twilio\Rest\Client;

class Notifier
{
    private Config $config;
    private Client $client;
    private string $messagingServiceSid;
    private bool $whatsappEnabled;

    public function __construct(Config $config)
    {
        $this->config = $config;
        $sid = $this->config->get('TWILIO_ACCOUNT_SID');
        $token = $this->config->get('TWILIO_AUTH_TOKEN');
        $this->messagingServiceSid = (string) $this->config->get('TWILIO_MESSAGING_SERVICE_SID');
        $this->whatsappEnabled = filter_var($this->config->get('TWILIO_WHATSAPP_ENABLED', 'false'), FILTER_VALIDATE_BOOLEAN);

        $this->client = new Client($sid, $token);
    }

    public function sendMessage(string $phoneNumber, string $messageBody): void
    {
        $to = $this->whatsappEnabled ? $this->formatWhatsApp($phoneNumber) : $phoneNumber;
        $this->client->messages->create($to, [
            'messagingServiceSid' => $this->messagingServiceSid,
            'body' => $messageBody,
        ]);
    }

    public function sendLink(string $phoneNumber, string $messageBody): void
    {
        $this->sendMessage($phoneNumber, $messageBody);
    }

    private function formatWhatsApp(string $phoneNumber): string
    {
        $trimmed = preg_replace('/\s+/', '', $phoneNumber);
        // PHP 7.4 compatibility: replace str_starts_with with strpos
        if (strpos($trimmed, 'whatsapp:') === 0) {
            return $trimmed;
        }

        return 'whatsapp:' . $trimmed;
    }
}
