<?php

namespace Parking;

use DateTimeImmutable;
use PDO;
use RuntimeException;

class MigrationRunner
{
    private Database $database;
    private Config $config;
    private string $migrationsPath;

    public function __construct(Database $database, Config $config, ?string $migrationsPath = null)
    {
        $this->database = $database;
        $this->config = $config;
        $this->migrationsPath = $migrationsPath ?? dirname(__DIR__) . '/database/migrations';
    }

    public function run(): void
    {
        $pdo = $this->database->connection();
        $driver = $pdo->getAttribute(PDO::ATTR_DRIVER_NAME);
        
        $this->ensureSchemaTable($pdo, $driver);

        // Use either embedded migrations or SQL files based on driver
        if ($driver === 'mysql') {
            $this->runMySQLMigrations($pdo);
        } else {
            $this->runSQLiteMigrations($pdo);
        }
    }

    private function ensureSchemaTable(PDO $pdo, string $driver): void
    {
        if ($driver === 'mysql') {
            $pdo->exec('CREATE TABLE IF NOT EXISTS schema_migrations (
                id INT AUTO_INCREMENT PRIMARY KEY,
                filename VARCHAR(255) NOT NULL UNIQUE,
                executed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci');
        } else {
            $pdo->exec('CREATE TABLE IF NOT EXISTS schema_migrations (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                filename TEXT NOT NULL UNIQUE,
                executed_at TEXT NOT NULL
            )');
        }
    }

    private function runMySQLMigrations(PDO $pdo): void
    {
        $executed = $this->fetchExecuted($pdo);

        // Prefer single-file full schema
        $fullFile = $this->migrationsPath . '/000_full_mysql.sql';
        $fullKey = '000_full_mysql.sql';

        // Compatibility: if older key recorded, treat as executed
        $alreadyExecuted = in_array($fullKey, $executed, true) || in_array('000_full_schema', $executed, true);

        if (!$alreadyExecuted) {
            try {
                if (is_file($fullFile)) {
                    $sql = trim(file_get_contents($fullFile) ?: '');
                    if ($sql !== '') {
                        $this->executeStatements($pdo, $sql);
                    }
                    $this->recordExecution($pdo, $fullKey);
                } else {
                    // Fallback to embedded schema if file missing
                    $this->executeStatements($pdo, $this->getMySQLFullSchema());
                    $this->recordExecution($pdo, '000_full_schema');
                }
            } catch (\Exception $e) {
                error_log("Migration error for full schema: " . $e->getMessage());
            }
        }

        // Idempotent ALTERs to align any existing databases
        $alters = [
            '006_alter_clients_add_fields' => $this->getMySQLAlterClientsSql($pdo),
            '007_alter_events_add_fields' => $this->getMySQLAlterEventsSql($pdo),
            '008_alter_coordinators_add_fields' => $this->getMySQLAlterCoordinatorsSql($pdo),
        ];

        foreach ($alters as $name => $sql) {
            if (in_array($name, $executed, true)) { continue; }
            try {
                if ($sql) { $this->executeStatements($pdo, $sql); }
                $this->recordExecution($pdo, $name);
            } catch (\Exception $e) {
                error_log("Migration error for $name: " . $e->getMessage());
            }
        }

        // Post-migration data fixes (idempotent)
        $this->runMySQLDataFixes($pdo);
    }

        private function getMySQLFullSchema(): string
        {
                // For fresh DBs: run once and record; idempotent due to IF NOT EXISTS and upserts
                return "
CREATE TABLE IF NOT EXISTS admins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    full_name VARCHAR(255) NOT NULL,
    email VARCHAR(255),
    role VARCHAR(50) DEFAULT 'admin',
    last_login_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

INSERT INTO admins (username, password_hash, full_name, email, role, created_at, updated_at)
VALUES ('admin', '\$2y\$10\$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Administrator', 'admin@example.com', 'admin', NOW(), NOW())
ON DUPLICATE KEY UPDATE username=username;

CREATE TABLE IF NOT EXISTS clients (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    logo VARCHAR(255) NULL,
    event_logo VARCHAR(255) NULL,
    primary_color VARCHAR(20) DEFAULT '#6366f1',
    secondary_color VARCHAR(20) DEFAULT '#10b981',
    event_name VARCHAR(255) NULL,
    event_date DATE NULL,
    event_location VARCHAR(255) NULL,
    contact_person VARCHAR(255) NULL,
    contact_phone VARCHAR(50) NULL,
    contact_email VARCHAR(255) NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

INSERT INTO clients (id, name, is_active, created_at, updated_at)
VALUES (1, 'Gulf Car', 1, NOW(), NOW())
ON DUPLICATE KEY UPDATE name=name;

CREATE TABLE IF NOT EXISTS events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    client_id INT NOT NULL,
    name VARCHAR(255) NOT NULL,
    logo VARCHAR(255) NULL,
    event_date DATE NULL,
    event_location VARCHAR(255) NULL,
    description TEXT,
    is_active TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    KEY idx_client_id (client_id),
    FOREIGN KEY (client_id) REFERENCES clients(id) ON DELETE CASCADE
) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS coordinators (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_id INT NULL,
    username VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    full_name VARCHAR(255) NOT NULL,
    phone VARCHAR(50) NULL,
    email VARCHAR(255),
    is_active TINYINT(1) DEFAULT 1,
    last_login_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    KEY idx_event_id (event_id),
    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE SET NULL
) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS tickets (
    id INT AUTO_INCREMENT PRIMARY KEY,
    customer_name VARCHAR(255) NOT NULL,
    phone_number VARCHAR(20) NOT NULL,
    license_plate VARCHAR(20) NOT NULL,
    valet_name VARCHAR(255) NOT NULL,
    event_id INT NULL,
    coordinator_id INT NULL,
    photos TEXT NULL,
    status VARCHAR(50) NOT NULL,
    request_token VARCHAR(255),
    request_url TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    requested_at TIMESTAMP NULL,
    returned_at TIMESTAMP NULL,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    KEY idx_tickets_status (status),
    KEY idx_tickets_token (request_token),
    KEY idx_event_id (event_id),
    KEY idx_coordinator_id (coordinator_id),
    CONSTRAINT fk_tickets_event FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE SET NULL,
    CONSTRAINT fk_tickets_coordinator FOREIGN KEY (coordinator_id) REFERENCES coordinators(id) ON DELETE SET NULL
) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
";
        }

    private function runSQLiteMigrations(PDO $pdo): void
    {
        $executed = $this->fetchExecuted($pdo);
        $files = $this->discoverMigrationFiles();

        foreach ($files as $file) {
            $name = basename($file);
            // Skip MySQL-only full schema file when on SQLite
            if ($name === '000_full_mysql.sql') {
                continue;
            }
            if (in_array($name, $executed, true)) {
                continue;
            }

            $sql = trim(file_get_contents($file) ?: '');
            if ($sql === '' || strpos($sql, '--') === 0) {
                $this->recordExecution($pdo, $name);
                continue;
            }

            try {
                $this->executeStatements($pdo, $sql);
                $this->recordExecution($pdo, $name);
            } catch (\Exception $e) {
                error_log("Migration error for $name: " . $e->getMessage());
            }
        }
    }

    private function getMySQLTicketsTable(): string
    {
        return 'CREATE TABLE IF NOT EXISTS tickets (
            id INT AUTO_INCREMENT PRIMARY KEY,
            customer_name VARCHAR(255) NOT NULL,
            phone_number VARCHAR(20) NOT NULL,
            license_plate VARCHAR(20) NOT NULL,
            valet_name VARCHAR(255) NOT NULL,
            event_id INT NULL,
            coordinator_id INT NULL,
            photos TEXT NULL,
            status VARCHAR(50) NOT NULL,
            request_token VARCHAR(255),
            request_url TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            requested_at TIMESTAMP NULL,
            returned_at TIMESTAMP NULL,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            KEY idx_tickets_status (status),
            KEY idx_tickets_token (request_token),
            KEY idx_event_id (event_id),
            KEY idx_coordinator_id (coordinator_id)
        ) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci';
    }

    private function getMySQLAdminsTable(): string
    {
        $adminPassword = '$2y$10$A1rnDPRy7fRdfgvuYDo2lukXf.sfzUtaL48NSggTV5d74rRbB9Nim'; // admin123
        
        return "CREATE TABLE IF NOT EXISTS admins (
            id INT AUTO_INCREMENT PRIMARY KEY,
            username VARCHAR(255) NOT NULL UNIQUE,
            password_hash VARCHAR(255) NOT NULL,
            full_name VARCHAR(255) NOT NULL,
            email VARCHAR(255),
            role VARCHAR(50) DEFAULT 'admin',
            last_login_at TIMESTAMP NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
        
        INSERT IGNORE INTO admins (username, password_hash, full_name, email, role, created_at, updated_at) 
        VALUES ('admin', '$adminPassword', 'Administrator', 'admin@example.com', 'admin', NOW(), NOW())";
    }

    private function getMySQLClientsTable(): string
    {
        return 'CREATE TABLE IF NOT EXISTS clients (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            logo VARCHAR(255) NULL,
            event_logo VARCHAR(255) NULL,
            primary_color VARCHAR(20) DEFAULT "#6366f1",
            secondary_color VARCHAR(20) DEFAULT "#10b981",
            event_name VARCHAR(255) NULL,
            event_date DATE NULL,
            event_location VARCHAR(255) NULL,
            contact_person VARCHAR(255) NULL,
            contact_phone VARCHAR(50) NULL,
            contact_email VARCHAR(255) NULL,
            is_active TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci';
    }

    private function getMySQLEventsTable(): string
    {
        return 'CREATE TABLE IF NOT EXISTS events (
            id INT AUTO_INCREMENT PRIMARY KEY,
            client_id INT NOT NULL,
            name VARCHAR(255) NOT NULL,
            logo VARCHAR(255) NULL,
            event_date DATE NULL,
            event_location VARCHAR(255) NULL,
            description TEXT,
            is_active TINYINT(1) DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            KEY idx_client_id (client_id),
            FOREIGN KEY (client_id) REFERENCES clients(id) ON DELETE CASCADE
        ) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci';
    }

    private function getMySQLCoordinatorsTable(): string
    {
        return 'CREATE TABLE IF NOT EXISTS coordinators (
            id INT AUTO_INCREMENT PRIMARY KEY,
            event_id INT NULL,
            username VARCHAR(255) NOT NULL UNIQUE,
            password_hash VARCHAR(255) NOT NULL,
            full_name VARCHAR(255) NOT NULL,
            phone VARCHAR(50) NULL,
            email VARCHAR(255),
            is_active TINYINT(1) DEFAULT 1,
            last_login_at TIMESTAMP NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            KEY idx_event_id (event_id),
            FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE SET NULL
        ) ENGINE=InnoDB CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci';
    }

    private function columnExists(PDO $pdo, string $table, string $column): bool
    {
        // Use SHOW COLUMNS instead of information_schema (blocked on some shared hosting)
        if (!preg_match('/^[A-Za-z0-9_]+$/', $table)) {
            return false;
        }
        try {
            $stmt = $pdo->prepare("SHOW COLUMNS FROM `{$table}` LIKE ?");
            $stmt->execute([$column]);
            return (bool) $stmt->fetchColumn();
        } catch (\Throwable $e) {
            return false;
        }
    }

    private function tableExists(PDO $pdo, string $table): bool
    {
        // Use SHOW TABLES instead of information_schema (blocked on some shared hosting)
        try {
            $stmt = $pdo->prepare('SHOW TABLES LIKE ?');
            $stmt->execute([$table]);
            return (bool) $stmt->fetchColumn();
        } catch (\Throwable $e) {
            return false;
        }
    }

    private function getMySQLAlterClientsSql(PDO $pdo): string
    {
        if (!$this->tableExists($pdo, 'clients')) {
            return '';
        }
        $alters = [];
        $add = function(string $col, string $def) use (&$alters, $pdo) {
            if (!$this->columnExists($pdo, 'clients', $col)) {
                $alters[] = "ADD COLUMN $col $def";
            }
        };
        $add('logo', 'VARCHAR(255) NULL');
        $add('event_logo', 'VARCHAR(255) NULL');
        $add('primary_color', 'VARCHAR(20) DEFAULT "#6366f1"');
        $add('secondary_color', 'VARCHAR(20) DEFAULT "#10b981"');
        $add('event_name', 'VARCHAR(255) NULL');
        $add('event_date', 'DATE NULL');
        $add('event_location', 'VARCHAR(255) NULL');
        $add('contact_person', 'VARCHAR(255) NULL');
        $add('contact_phone', 'VARCHAR(50) NULL');
        $add('contact_email', 'VARCHAR(255) NULL');
        $add('is_active', 'TINYINT(1) DEFAULT 1');
        if (!$alters) { return ''; }
        return 'ALTER TABLE clients ' . implode(', ', $alters);
    }

    private function getMySQLAlterEventsSql(PDO $pdo): string
    {
        if (!$this->tableExists($pdo, 'events')) {
            return '';
        }
        $alters = [];
        $add = function(string $col, string $def) use (&$alters, $pdo) {
            if (!$this->columnExists($pdo, 'events', $col)) {
                $alters[] = "ADD COLUMN $col $def";
            }
        };
        // If old schema had ticket_id/event_type, we won't drop them here; only add required fields
        $add('client_id', 'INT NULL');
        $add('name', 'VARCHAR(255) NULL');
        $add('logo', 'VARCHAR(255) NULL');
        $add('event_date', 'DATE NULL');
        $add('event_location', 'VARCHAR(255) NULL');
        $add('description', 'TEXT NULL');
        $add('is_active', 'TINYINT(1) DEFAULT 0');
        $add('updated_at', 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP');
        if (!$alters) { return ''; }
        return 'ALTER TABLE events ' . implode(', ', $alters);
    }

    private function getMySQLAlterCoordinatorsSql(PDO $pdo): string
    {
        if (!$this->tableExists($pdo, 'coordinators')) {
            return '';
        }
        $alters = [];
        $add = function(string $col, string $def) use (&$alters, $pdo) {
            if (!$this->columnExists($pdo, 'coordinators', $col)) {
                $alters[] = "ADD COLUMN $col $def";
            }
        };
        $add('event_id', 'INT NULL');
        $add('phone', 'VARCHAR(50) NULL');
        $add('is_active', 'TINYINT(1) DEFAULT 1');
        if (!$alters) { return ''; }
        return 'ALTER TABLE coordinators ' . implode(', ', $alters);
    }

    private function fetchExecuted(PDO $pdo): array
    {
        try {
            $stmt = $pdo->query('SELECT filename FROM schema_migrations');
            $rows = $stmt ? $stmt->fetchAll(PDO::FETCH_COLUMN) : [];
            return $rows ?: [];
        } catch (\Exception $e) {
            return [];
        }
    }

    private function discoverMigrationFiles(): array
    {
        if (!is_dir($this->migrationsPath)) {
            throw new RuntimeException('لم يتم العثور على مجلد الهجرات: ' . $this->migrationsPath);
        }

        $files = glob($this->migrationsPath . '/*.sql');
        sort($files, SORT_NATURAL);
        return $files;
    }

    private function executeStatements(PDO $pdo, string $sql): void
    {
        // Strip SQL line comments (-- ...) before splitting
        $lines = explode("\n", $sql);
        $cleaned = [];
        foreach ($lines as $line) {
            $trimmed = ltrim($line);
            if ($trimmed === '' || strpos($trimmed, '--') === 0) {
                continue; // skip empty lines and comment-only lines
            }
            $cleaned[] = $line;
        }
        $sql = implode("\n", $cleaned);

        // Split by semicolon
        $statements = array_filter(array_map('trim', explode(';', $sql)), function ($s) {
            return $s !== '';
        });
        foreach ($statements as $stmt) {
            $pdo->exec($stmt);
        }
    }

    private function recordExecution(PDO $pdo, string $filename): void
    {
        try {
            // Try MySQL syntax first
            $stmt = $pdo->prepare('INSERT INTO schema_migrations (filename) VALUES (?)');
            $stmt->execute([$filename]);
        } catch (\Exception $e) {
            // Fallback to SQLite syntax
            try {
                $stmt = $pdo->prepare('INSERT INTO schema_migrations (filename, executed_at) VALUES (?, ?)');
                $stmt->execute([$filename, (new DateTimeImmutable())->format('Y-m-d H:i:s')]);
            } catch (\Exception $ex) {
                error_log("Failed to record migration: " . $ex->getMessage());
            }
        }
    }

    private function runMySQLDataFixes(PDO $pdo): void
    {
        try {
            // Backfill coordinators.password_hash from password if exists and hash column is null
            $hasPwd = $this->columnExists($pdo, 'coordinators', 'password');
            $hasPwdHash = $this->columnExists($pdo, 'coordinators', 'password_hash');
            if ($hasPwd && $hasPwdHash) {
                $pdo->exec("UPDATE coordinators SET password_hash = password WHERE password_hash IS NULL AND password IS NOT NULL");
            }
        } catch (\Throwable $e) {
            // ignore
        }
    }
}
