<?php

namespace Parking;

use PDO;
use PDOException;
use Throwable;

/**
 * Gulf Car Parking — Installation & Update Manager
 *
 * Handles:
 *   - Fresh install (write .env + create tables)
 *   - Update / repair (run pending migrations, preserve data)
 *   - Status detection (safe, never throws)
 *   - DB connection testing before writing config
 *   - Auto URL detection for localhost & hosting
 */
class Installer
{
    private string $rootPath;

    public function __construct(?string $rootPath = null)
    {
        $this->rootPath = $rootPath ?? dirname(__DIR__);
    }

    /* ─── Path Helpers ──────────────────────────────────────────── */

    public function envPath(): string
    {
        return $this->rootPath . DIRECTORY_SEPARATOR . '.env';
    }

    public function envExists(): bool
    {
        return file_exists($this->envPath());
    }

    /* ─── Auto-detect APP_URL ───────────────────────────────────── */

    /**
     * Build the public-facing URL from the current HTTP request.
     * install.php is inside /public/ so dirname() gives the correct base.
     *
     * localhost  →  http://localhost/parking/public
     * hosting    →  http://park.meezmart.com/public   (or just / if doc-root = public)
     */
    public static function detectAppUrl(): string
    {
        $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host  = $_SERVER['HTTP_HOST'] ?? $_SERVER['SERVER_NAME'] ?? 'localhost';
        $script = str_replace('\\', '/', $_SERVER['SCRIPT_NAME'] ?? '/install.php');
        $dir   = rtrim(dirname($script), '/');
        return $proto . '://' . $host . $dir;
    }

    /* ─── Requirements Check ────────────────────────────────────── */

    public function checkRequirements(): array
    {
        $pdoMysql  = extension_loaded('pdo_mysql');
        $writable  = is_writable($this->rootPath);

        return [
            [
                'label' => 'PHP 7.4 أو أحدث',
                'pass'  => version_compare(PHP_VERSION, '7.4.0', '>='),
                'hint'  => 'الإصدار الحالي: ' . PHP_VERSION,
            ],
            [
                'label' => 'PDO Extension',
                'pass'  => extension_loaded('pdo'),
                'hint'  => 'مطلوب للاتصال بقاعدة البيانات',
            ],
            [
                'label' => 'دعم MySQL (pdo_mysql)',
                'pass'  => $pdoMysql,
                'hint'  => $pdoMysql ? 'متوفر ✓' : 'غير متوفر — يجب تفعيله',
            ],
            [
                'label' => 'صلاحيات الكتابة',
                'pass'  => $writable,
                'hint'  => $writable ? 'متوفرة ✓' : 'غير متوفرة — مطلوبة لكتابة ملف .env',
            ],
        ];
    }

    /* ─── Safe Status Detection ─────────────────────────────────── */

    /**
     * Detect system state without throwing any exceptions.
     *
     * Returns array with keys:
     *   env_exists   – bool
     *   db_connected – bool
     *   tables_exist – bool
     *   admin_exists – bool
     *   mode         – 'fresh_install' | 'needs_migration' | 'installed'
     *   error        – string|null
     */
    public function getStatus(?Database $database = null): array
    {
        $s = [
            'env_exists'   => $this->envExists(),
            'db_connected' => false,
            'tables_exist' => false,
            'admin_exists' => false,
            'mode'         => 'fresh_install',
            'error'        => null,
        ];

        if (!$s['env_exists'] || !$database) {
            return $s;
        }

        /* Try connecting */
        try {
            $pdo = $database->connection();
            $s['db_connected'] = true;
        } catch (Throwable $e) {
            $s['error'] = $e->getMessage();
            return $s;
        }

        /* Check tables */
        try {
            $s['tables_exist'] = $this->hasTable($pdo, 'tickets');

            if ($s['tables_exist']) {
                $s['admin_exists'] = $this->hasTable($pdo, 'admins')
                    && (int) $pdo->query('SELECT COUNT(*) FROM admins')->fetchColumn() > 0;
                $s['mode'] = 'installed';
            } else {
                $s['mode'] = 'needs_migration';
            }
        } catch (Throwable $e) {
            $s['error'] = $e->getMessage();
            $s['mode'] = 'needs_migration';
        }

        return $s;
    }

    /* ─── Is System Installed? (safe, never throws) ─────────────── */

    public function isInstalled(Database $database): bool
    {
        if (!$this->envExists()) {
            return false;
        }
        try {
            return $this->hasTable($database->connection(), 'tickets');
        } catch (Throwable $e) {
            return false;
        }
    }

    /* ─── Test DB Connection (raw params, no .env needed) ──────── */

    /**
     * Test a MySQL connection with raw form-submitted values.
     * If the database doesn't exist, attempt to CREATE it.
     *
     * @return array{success: bool, message: string}
     */
    public static function testDbConnection(array $p): array
    {
        $host    = $p['host']    ?? 'localhost';
        $port    = $p['port']    ?? '3306';
        $name    = $p['name']    ?? '';
        $user    = $p['user']    ?? 'root';
        $pass    = $p['password'] ?? '';
        $charset = $p['charset'] ?? 'utf8mb4';

        if ($name === '') {
            return ['success' => false, 'message' => 'اسم قاعدة البيانات مطلوب'];
        }

        /* 1. Try connecting WITH the database name */
        try {
            new PDO(
                "mysql:host={$host};port={$port};dbname={$name};charset={$charset}",
                $user, $pass,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            return ['success' => true, 'message' => "اتصال ناجح بقاعدة البيانات «{$name}»"];
        } catch (PDOException $e) {
            /* 2. Unknown database → create it */
            if (strpos($e->getMessage(), '1049') !== false) {
                try {
                    $pdo = new PDO(
                        "mysql:host={$host};port={$port};charset={$charset}",
                        $user, $pass,
                        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                    );
                    $safe = str_replace('`', '``', $name);
                    $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$safe}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                    return ['success' => true, 'message' => "تم إنشاء قاعدة البيانات «{$name}» بنجاح"];
                } catch (Throwable $e2) {
                    return ['success' => false, 'message' => 'فشل إنشاء قاعدة البيانات: ' . $e2->getMessage()];
                }
            }
            /* 3. Access denied or other error */
            if (strpos($e->getMessage(), '1045') !== false) {
                return ['success' => false, 'message' => 'خطأ في بيانات الاتصال: اسم المستخدم أو كلمة المرور غير صحيحة'];
            }
            return ['success' => false, 'message' => 'فشل الاتصال: ' . $e->getMessage()];
        }
    }

    /* ─── Load Current .env Values ──────────────────────────────── */

    public function loadCurrentEnv(): array
    {
        if (!$this->envExists()) {
            return [];
        }

        $config = new Config($this->envPath());

        $keys = [
            'APP_ENV', 'APP_URL',
            'DB_DRIVER', 'DB_HOST', 'DB_PORT', 'DB_NAME', 'DB_USER', 'DB_PASSWORD', 'DB_CHARSET',
            'DB_PATH', 'MIGRATE_TOKEN',
            'TWILIO_ACCOUNT_SID', 'TWILIO_AUTH_TOKEN', 'TWILIO_MESSAGING_SERVICE_SID',
            'TWILIO_WHATSAPP_ENABLED', 'ATTENDANT_PHONE',
        ];

        $result = [];
        foreach ($keys as $key) {
            $result[$key] = $config->get($key, '');
        }
        return $result;
    }

    /* ─── Write .env (fresh install only) ───────────────────────── */

    public function writeEnv(array $values): void
    {
        $lines = [];
        foreach ($values as $key => $value) {
            $lines[] = $key . '=' . $this->formatEnvValue((string) $value);
        }

        if (@file_put_contents($this->envPath(), implode("\n", $lines) . "\n") === false) {
            throw new \RuntimeException('فشل كتابة ملف .env — تحقق من صلاحيات الكتابة على المجلد');
        }
    }

    /* ─── Run Migrations (idempotent — safe for install & update) ─ */

    /**
     * @return array{messages: string[], errors: string[]}
     */
    public function runMigrations(Database $database): array
    {
        $messages = [];
        $errors   = [];

        try {
            $config = new Config();
            $runner = new MigrationRunner(
                $database,
                $config,
                $this->rootPath . '/database/migrations'
            );
            $runner->run();
            $messages[] = 'تم تشغيل الهجرات وإنشاء/تحديث الجداول بنجاح';
        } catch (Throwable $e) {
            $errors[] = 'خطأ في الهجرات: ' . $e->getMessage();
        }

        return ['messages' => $messages, 'errors' => $errors];
    }

    /* ─── Table Existence Check (no information_schema) ─────────── */

    private function hasTable(PDO $pdo, string $table): bool
    {
        try {
            $driver = $pdo->getAttribute(PDO::ATTR_DRIVER_NAME);

            if ($driver === 'mysql') {
                $stmt = $pdo->prepare('SHOW TABLES LIKE ?');
                $stmt->execute([$table]);
                return (bool) $stmt->fetchColumn();
            }

            // SQLite
            $stmt = $pdo->prepare("SELECT name FROM sqlite_master WHERE type='table' AND name=? LIMIT 1");
            $stmt->execute([$table]);
            return (bool) $stmt->fetchColumn();
        } catch (Throwable $e) {
            return false;
        }
    }

    /* ─── .env Value Formatter ──────────────────────────────────── */

    private function formatEnvValue(string $value): string
    {
        if ($value === '') {
            return '';
        }
        // Wrap in quotes if value contains special characters
        if (preg_match('/[\s#="\'\\\\]/', $value)) {
            return '"' . str_replace('"', '\\"', $value) . '"';
        }
        return $value;
    }
}
