<?php

namespace Parking;

class Config
{
    private array $values = [];

    public function __construct(?string $envFile = null)
    {
        $root = dirname(__DIR__);

        // 1. Load base .env file (production/default config)
        $envFile ??= $root . '/.env';
        if (file_exists($envFile)) {
            $this->values = array_merge($this->values, $this->parse(file_get_contents($envFile)));
        }

        // 2. Load .env.local if it exists — overrides .env (used for local development)
        //    .env.local should be added to .gitignore and never committed
        $localEnvFile = $root . '/.env.local';
        if (file_exists($localEnvFile)) {
            $this->values = array_merge($this->values, $this->parse(file_get_contents($localEnvFile)));
        }

        // 3. Merge with actual environment variables (highest priority)
        foreach ($_ENV as $key => $value) {
            $this->values[$key] = $value;
        }
        foreach ($_SERVER as $key => $value) {
            if (is_string($value) && !array_key_exists($key, $this->values)) {
                $this->values[$key] = $value;
            }
        }
    }

    public function get(string $key, ?string $default = null): ?string
    {
        return $this->values[$key] ?? $default;
    }

    private function parse(string $content): array
    {
        $lines = preg_split('/\r?\n/', $content);
        $result = [];
        foreach ($lines as $line) {
            $trimmed = trim($line);
            // PHP 7.4 compatibility: replace str_starts_with() with strpos check
            if ($trimmed === '' || strpos($trimmed, '#') === 0) {
                continue;
            }

            // PHP 7.4 compatibility: replace str_contains() with strpos check
            if (strpos($trimmed, '=') === false) {
                continue;
            }

            [$key, $value] = array_map('trim', explode('=', $trimmed, 2));
            $value = trim($value, " \"'\t");
            if ($value === 'true') {
                $value = '1';
            } elseif ($value === 'false') {
                $value = '0';
            }
            $result[$key] = $value;
        }

        return $result;
    }
}
