<?php

namespace Parking;

use PDO;

class ClientRepository
{
    private Database $database;

    public function __construct(Database $database)
    {
        $this->database = $database;
        if (session_status() === PHP_SESSION_NONE) {
            @session_start();
        }
    }

    public function getActiveClient(): ?array
    {
        $pdo = $this->database->connection();

        if (isset($_SESSION['client_id'])) {
            $stmt = $pdo->prepare('SELECT * FROM clients WHERE id = ? AND is_active = 1');
            $stmt->execute([$_SESSION['client_id']]);
            $client = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($client) {
                return $client;
            }
        }

        $stmt = $pdo->query('SELECT * FROM clients WHERE is_active = 1 ORDER BY id LIMIT 1');
        $client = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
        if ($client) {
            $_SESSION['client_id'] = (int)$client['id'];
        }
        return $client;
    }

    public function getAllClients(): array
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->query('SELECT * FROM clients ORDER BY created_at DESC');
        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public function getClientById(int $id): ?array
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare('SELECT * FROM clients WHERE id = ?');
        $stmt->execute([$id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function createClient(array $data): bool
    {
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare(
            'INSERT INTO clients (
                name, logo, event_logo, primary_color, secondary_color,
                event_name, event_date, event_location,
                contact_person, contact_phone, contact_email
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)'
        );
        return (bool) $stmt->execute([
            $data['name'],
            $data['logo'] ?? null,
            $data['event_logo'] ?? null,
            $data['primary_color'] ?? '#6366f1',
            $data['secondary_color'] ?? '#10b981',
            $data['event_name'] ?? null,
            $data['event_date'] ?? null,
            $data['event_location'] ?? null,
            $data['contact_person'] ?? null,
            $data['contact_phone'] ?? null,
            $data['contact_email'] ?? null,
        ]);
    }

    public function updateClient(int $id, array $data): bool
    {
        $pdo = $this->database->connection();
        $driver = $pdo->getAttribute(PDO::ATTR_DRIVER_NAME);
        $nowExpr = ($driver === 'mysql') ? 'NOW()' : "datetime('now')";

        $sql = "UPDATE clients SET
                name = ?,
                logo = ?,
                event_logo = ?,
                primary_color = ?,
                secondary_color = ?,
                event_name = ?,
                event_date = ?,
                event_location = ?,
                contact_person = ?,
                contact_phone = ?,
                contact_email = ?,
                updated_at = $nowExpr
            WHERE id = ?";

        $stmt = $pdo->prepare($sql);
        return (bool) $stmt->execute([
            $data['name'],
            $data['logo'] ?? null,
            $data['event_logo'] ?? null,
            $data['primary_color'] ?? '#6366f1',
            $data['secondary_color'] ?? '#10b981',
            $data['event_name'] ?? null,
            $data['event_date'] ?? null,
            $data['event_location'] ?? null,
            $data['contact_person'] ?? null,
            $data['contact_phone'] ?? null,
            $data['contact_email'] ?? null,
            $id,
        ]);
    }

    public function toggleClientStatus(int $id): bool
    {
        $pdo = $this->database->connection();
        $driver = $pdo->getAttribute(PDO::ATTR_DRIVER_NAME);
        $nowExpr = ($driver === 'mysql') ? 'NOW()' : "datetime('now')";

        $stmt = $pdo->prepare(
            "UPDATE clients
             SET is_active = CASE WHEN is_active = 1 THEN 0 ELSE 1 END,
                 updated_at = $nowExpr
             WHERE id = ?"
        );
        return (bool) $stmt->execute([$id]);
    }

    public function deleteClient(int $id): bool
    {
        if ($id === 1) {
            // keep default seeded client
            return false;
        }
        $pdo = $this->database->connection();
        $stmt = $pdo->prepare('DELETE FROM clients WHERE id = ?');
        return (bool) $stmt->execute([$id]);
    }

    public function switchClient(int $id): bool
    {
        $client = $this->getClientById($id);
        if ($client && (!isset($client['is_active']) || (int)$client['is_active'] === 1)) {
            $_SESSION['client_id'] = $id;
            return true;
        }
        return false;
    }
}
