<?php

declare(strict_types=1);

// Log errors instead of displaying them to prevent HTML in SSE stream
ini_set('display_errors', '0');
ini_set('log_errors', '1');
error_reporting(E_ALL);

use Parking\Installer;
use Parking\ParkingRepository;
use Parking\CoordinatorRepository;

try {
    // Start session to check user type
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    ignore_user_abort(true);
    set_time_limit(0);

    $container = require __DIR__ . '/../bootstrap.php';
    $database = $container['database'];

    // Get coordinator info if logged in
    $coordinatorEventId = null;
    $coordinatorId = null;
    $isValidSession = false;

    if (isset($_SESSION['admin_id'])) {
        // Admin is logged in
        $isValidSession = true;
    } elseif (isset($_SESSION['coordinator_id'])) {
        // Coordinator is logged in - verify they exist
        $coordinatorRepo = new CoordinatorRepository($database);
        $coordinator = $coordinatorRepo->getCoordinatorById($_SESSION['coordinator_id']);
        
        if ($coordinator) {
            $coordinatorEventId = $coordinator['event_id'] !== null ? (int)$coordinator['event_id'] : null;
            $coordinatorId = $coordinator['id'] !== null ? (int)$coordinator['id'] : null;
            $isValidSession = true;
        }
    }

    // Close session to prevent locking
    session_write_close();
} catch (Throwable $e) {
    header('Content-Type: text/event-stream; charset=utf-8');
    header('Cache-Control: no-cache');
    echo "event: error\n";
    echo 'data: {"error": "initialization_failed", "message": "خطأ في تهيئة الاتصال"}'."\n\n";
    exit;
}

// If no valid session, return error
if (!$isValidSession) {
    header('Content-Type: text/event-stream; charset=utf-8');
    header('Cache-Control: no-cache');
    echo "event: error\n";
    echo 'data: {"error": "session_expired", "message": "جلستك منتهية. يرجى تسجيل الدخول مرة أخرى."}'."\n\n";
    exit;
}

$installer = new Installer(dirname(__DIR__));

if (!$installer->isInstalled($database)) {
    header('HTTP/1.1 503 Service Unavailable');
    exit;
}

header('Content-Type: text/event-stream; charset=utf-8');
header('Cache-Control: no-cache');
header('X-Accel-Buffering: no');

if (function_exists('apache_setenv')) {
    @apache_setenv('no-gzip', '1');
}

while (ob_get_level() > 0) {
    ob_end_flush();
}

$repository = new ParkingRepository($database);
$lastChecksum = null;
$start = time();
$maxDuration = 300; // 5 minutes

while (!connection_aborted() && (time() - $start) < $maxDuration) {
    try {
        // Coordinator sees only their tickets, Admin sees all
        $tickets = $repository->getActiveTickets($coordinatorEventId, $coordinatorId);
    $payload = json_encode([
        'timestamp' => time(),
        'tickets' => $tickets,
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    if ($payload === false) {
        $payload = json_encode(['error' => 'encoding_failed', 'timestamp' => time()]);
    }

        $checksum = md5((string) $payload);

        if ($checksum !== $lastChecksum) {
            echo "event: tickets\n";
            echo 'data: ' . $payload . "\n\n";
            $lastChecksum = $checksum;
        } else {
            echo "event: ping\n";
            echo 'data: {"timestamp":' . time() . "}\n\n";
        }

        @ob_flush();
        @flush();

    } catch (Exception $e) {
        // Send error event but continue trying
        echo "event: error\n";
        echo 'data: {"error":"query_failed","message":"خطأ مؤقت في الاتصال"}' . "\n\n";
        @ob_flush();
        @flush();
    }

    sleep(3);
}

echo "event: close\n";
echo 'data: {"reason":"timeout"}' . "\n\n";
@ob_flush();
@flush();
