<?php
require_once __DIR__ . '/session_init.php';

use Parking\Installer;
use Parking\EventRepository;
use Parking\CoordinatorRepository;
use Parking\Auth;

$container = require __DIR__ . '/../bootstrap.php';
$config = $container['config'];
$database = $container['database'];

$installer = new Installer(dirname(__DIR__));

if (!$installer->isInstalled($database)) {
    header('Location: install.php');
    exit;
}

// Authentication: Only Admin or Coordinator can access
$isAdmin = false;
$isCoordinator = false;

if (isset($_SESSION['admin_id'])) {
    $auth = new Auth($database);
    $auth->requireAuth();
    $isAdmin = true;
} elseif (isset($_SESSION['coordinator_id'])) {
    $coordinatorRepo = new CoordinatorRepository($database);
    $coordinator = $coordinatorRepo->getCoordinatorById($_SESSION['coordinator_id']);
    if (!$coordinator) {
        session_destroy();
        header('Location: coordinator-login.php');
        exit;
    }
    $isCoordinator = true;
} else {
    // Not logged in - redirect to login
    header('Location: login.php');
    exit;
}

$eventRepo = new EventRepository($database);

// Check if coordinator is logged in
$coordinatorEventId = null;
if ($isCoordinator) {
    $coordinatorEventId = $coordinator['event_id'];
}

// Get active event - if coordinator is logged in, get their specific event
if ($coordinatorEventId) {
    $activeEvent = $eventRepo->getEventById($coordinatorEventId);
} else {
    $activeEvent = $eventRepo->getActiveEvent();
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تسجيل سيارة - نظام الباركنج</title>
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/checkin.css">
</head>
<body class="rtl">
            <div class="main-wrap">
                <div class="brand-bar" style="margin-bottom:12px;">
                    <div style="width: 36px; height: 36px; background: var(--brand-gradient); border-radius: 10px; display: flex; align-items: center; justify-content: center; flex-shrink: 0;">
                        <span class="material-icons" style="font-size: 22px; color: #fff;">local_parking</span>
                    </div>
                    <span class="brand-name">شركة الخليج</span>
                </div>
                
                <!-- Event Banner -->
                <?php if ($activeEvent): ?>
                    <div style="background: linear-gradient(135deg, rgba(108, 99, 255, 0.08) 0%, rgba(0, 217, 166, 0.06) 100%); border: 1px solid rgba(108, 99, 255, 0.15); border-radius: 14px; padding: 12px 20px; margin-bottom: 16px; display: flex; align-items: center; gap: 16px;">
                        <?php if (!empty($activeEvent['logo'])): ?>
                            <div style="flex-shrink: 0;">
                                <img src="<?php echo htmlspecialchars($activeEvent['logo']); ?>" alt="Event Logo" style="height: 50px; width: auto; max-width: 100px; object-fit: contain; filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));" />
                            </div>
                        <?php endif; ?>
                        
                        <div style="flex: 1; min-width: 0;">
                            <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 4px;">
                                <span class="material-icons" style="font-size: 18px; color: var(--primary);">event</span>
                                <h3 style="font-size: 16px; font-weight: 700; color: var(--primary); margin: 0;">
                                    <?php echo htmlspecialchars($activeEvent['name']); ?>
                                </h3>
                            </div>
                            
                            <div style="display: flex; flex-wrap: wrap; gap: 12px; font-size: 11px; color: var(--gray);">
                                <?php if ($activeEvent['event_date']): ?>
                                    <div style="display: flex; align-items: center; gap: 4px;">
                                        <span class="material-icons" style="font-size: 14px; color: var(--primary);">calendar_today</span>
                                        <span><?php echo htmlspecialchars($activeEvent['event_date']); ?></span>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($activeEvent['event_location']): ?>
                                    <div style="display: flex; align-items: center; gap: 4px;">
                                        <span class="material-icons" style="font-size: 14px; color: var(--success);">location_on</span>
                                        <span><?php echo htmlspecialchars($activeEvent['event_location']); ?></span>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                
                <div class="header">
                    <div class="title">إدخال سيارة جديدة</div>
                <div class="actions">
                    <a class="btn" href="dashboard.php"><span class="material-icons">monitor</span> لوحة المتابعة</a>
                </div>
            </div>
            <div class="grid">
                <section class="card">
                    <form method="post" action="register.php" enctype="multipart/form-data">
                        <?php if ($activeEvent): ?>
                            <input type="hidden" name="event_id" value="<?php echo htmlspecialchars($activeEvent['id']); ?>">
                        <?php endif; ?>
                        <div class="form-grid">
                            <div>
                                <label>اسم العميل</label>
                                <input class="input" type="text" name="customer_name" required autofocus>
                            </div>
                            <div>
                                <label>رقم الهاتف</label>
                                <input class="input" type="tel" name="phone_number" placeholder="249XXXXXXXXX" required>
                            </div>
                            <div>
                                <label>رقم اللوحة</label>
                                <input class="input" type="text" name="license_plate" required>
                            </div>
                            <div>
                                <label>اسم السائق</label>
                                <input class="input" type="text" name="valet_name" required>
                            </div>
                            
                            <!-- Photos Upload Section -->
                            <div class="full">
                                <label style="display: flex; align-items: center; gap: 8px; margin-bottom: 8px;">
                                    <span class="material-icons" style="color: var(--primary);">photo_camera</span>
                                    صور السيارة (اختياري)
                                </label>
                                
                                <!-- Hidden file input -->
                                <input type="file" id="photoInput" name="photos[]" accept="image/*" capture="environment" multiple style="display: none;">
                                
                                <!-- Custom upload button -->
                                <button type="button" class="btn" id="addPhotoBtn" style="width: 100%; margin-bottom: 12px; background: rgba(108, 99, 255, 0.1); border: 2px dashed var(--primary); color: var(--primary);">
                                    <span class="material-icons">add_photo_alternate</span>
                                    التقط أو اختر صور السيارة
                                </button>
                                
                                <!-- Photos preview container -->
                                <div id="photosPreview" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(120px, 1fr)); gap: 12px; margin-top: 12px;"></div>
                            </div>
                            
                            <div class="full">
                                <div class="btn-row">
                                      <button class="btn btn-primary" type="submit"><span class="material-icons">add_circle</span> تسجيل السيارة</button>
                                    <a class="btn" href="dashboard.php"><span class="material-icons">dashboard</span> فتح لوحة المتابعة</a>
                                </div>
                            </div>
                        </div>
                        <p class="tip">بعد التسجيل سيتم إرسال رابط للعميل لطلب السيارة عند الحاجة.</p>
                    </form>
                </section>
                <section class="card">
                    <h2 style="margin-top:0">تلميحات</h2>
                    <ul style="margin:0; padding-right:18px; color: var(--muted); line-height: 1.9;">
                        <li>تأكد من إدخال رقم الهاتف بصيغة صحيحة.</li>
                        <li>استخدم اسم السائق كما سيظهر في لوحة المتابعة.</li>
                        <li>يمكنك الانتقال للشاشة الحية لمتابعة الحالات فوراً.</li>
                        <li>يمكنك التقاط عدة صور للسيارة من زوايا مختلفة.</li>
                    </ul>
                </section>
            </div>
        </div>
    
    <script>
        // Photo upload functionality
        const photoInput = document.getElementById('photoInput');
        const addPhotoBtn = document.getElementById('addPhotoBtn');
        const photosPreview = document.getElementById('photosPreview');
        let selectedFiles = [];

        // Trigger file input when button is clicked
        addPhotoBtn.addEventListener('click', () => {
            photoInput.click();
        });

        // Handle file selection
        photoInput.addEventListener('change', (e) => {
            const files = Array.from(e.target.files);
            
            files.forEach((file, index) => {
                if (file.type.startsWith('image/')) {
                    selectedFiles.push(file);
                    displayPhotoPreview(file, selectedFiles.length - 1);
                }
            });
            
            // Reset input to allow selecting same files again
            photoInput.value = '';
        });

        function displayPhotoPreview(file, index) {
            const reader = new FileReader();
            
            reader.onload = (e) => {
                const photoCard = document.createElement('div');
                photoCard.style.cssText = 'position: relative; border-radius: 8px; overflow: hidden; background: var(--card); border: 2px solid rgba(99, 102, 241, 0.2); aspect-ratio: 1;';
                
                photoCard.innerHTML = `
                    <img src="${e.target.result}" style="width: 100%; height: 100%; object-fit: cover;">
                    <button type="button" class="remove-photo-btn" data-index="${index}" 
                            style="position: absolute; top: 4px; left: 4px; background: rgba(239, 68, 68, 0.9); 
                                   color: white; border: none; border-radius: 50%; width: 28px; height: 28px; 
                                   cursor: pointer; display: flex; align-items: center; justify-content: center;
                                   box-shadow: 0 2px 8px rgba(0,0,0,0.3); transition: all 0.2s;">
                        <span class="material-icons" style="font-size: 18px;">close</span>
                    </button>
                `;
                
                photosPreview.appendChild(photoCard);
                
                // Add hover effect
                const removeBtn = photoCard.querySelector('.remove-photo-btn');
                removeBtn.addEventListener('mouseenter', () => {
                    removeBtn.style.transform = 'scale(1.1)';
                });
                removeBtn.addEventListener('mouseleave', () => {
                    removeBtn.style.transform = 'scale(1)';
                });
                
                // Remove photo
                removeBtn.addEventListener('click', () => {
                    removePhoto(index, photoCard);
                });
            };
            
            reader.readAsDataURL(file);
        }

        function removePhoto(index, photoCard) {
            selectedFiles.splice(index, 1);
            photoCard.remove();
            updatePhotosInput();
        }

        function updatePhotosInput() {
            // Create a new DataTransfer to update the file input
            const dataTransfer = new DataTransfer();
            selectedFiles.forEach(file => {
                dataTransfer.items.add(file);
            });
            photoInput.files = dataTransfer.files;
        }

        // Form submission - show loading state
        document.querySelector('form').addEventListener('submit', function(e) {
            const submitBtn = this.querySelector('button[type="submit"]');
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<span class="material-icons">hourglass_empty</span> جاري التسجيل...';
        });
    </script>
</body>
</html>
