<?php
// Direct Installation Script
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Direct Installation</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 50px auto; padding: 20px; }
        .success { color: green; background: #d4edda; padding: 10px; border-radius: 5px; margin: 10px 0; }
        .error { color: red; background: #f8d7da; padding: 10px; border-radius: 5px; margin: 10px 0; }
        .info { color: blue; background: #d1ecf1; padding: 10px; border-radius: 5px; margin: 10px 0; }
        pre { background: #f5f5f5; padding: 10px; border-radius: 5px; overflow-x: auto; }
        .btn { display: inline-block; padding: 10px 20px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; margin: 10px 0; }
        .btn:hover { background: #0056b3; }
    </style>
</head>
<body>";

echo "<h1>🚀 Direct Database Installation</h1>";

try {
    // Load bootstrap
    $container = require __DIR__ . '/../bootstrap.php';
    $database = $container['database'];
    $config = $container['config'];
    
    echo "<div class='success'>✓ Bootstrap loaded successfully</div>";
    
    // Get database connection
    $pdo = $database->connection();
    echo "<div class='success'>✓ Connected to database: " . $config->get('DB_NAME') . "</div>";
    
    // Check if already installed
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE 'tickets'");
        $exists = $stmt->fetchColumn();
        
        if ($exists) {
            echo "<div class='info'>⚠️ Database tables already exist!</div>";
            echo "<p><a href='index.php' class='btn'>Go to Home Page</a></p>";
            exit;
        }
    } catch (Exception $e) {
        // Table doesn't exist, continue with installation
    }
    
    echo "<h2>📋 Installing Database Schema...</h2>";
    
    // Read SQL file
    $sqlFile = dirname(__DIR__) . '/database/migrations/000_full_mysql.sql';
    
    if (!file_exists($sqlFile)) {
        throw new Exception("SQL migration file not found: $sqlFile");
    }
    
    $sql = file_get_contents($sqlFile);
    echo "<div class='success'>✓ SQL file loaded: " . basename($sqlFile) . "</div>";
    
    // Split SQL into individual statements
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($stmt) {
            return !empty($stmt) && stripos($stmt, 'CREATE') !== false || 
                   stripos($stmt, 'INSERT') !== false || 
                   stripos($stmt, 'ALTER') !== false;
        }
    );
    
    echo "<div class='info'>Found " . count($statements) . " SQL statements to execute</div>";
    
    // Execute each statement
    $successCount = 0;
    $errorCount = 0;
    $errors = [];
    
    foreach ($statements as $index => $statement) {
        try {
            $pdo->exec($statement . ';');
            $successCount++;
            
            // Show what was created
            if (preg_match('/CREATE TABLE [IF NOT EXISTS ]*`?(\w+)`?/i', $statement, $matches)) {
                echo "<div class='success'>✓ Created table: {$matches[1]}</div>";
            }
        } catch (PDOException $e) {
            $errorCount++;
            $errors[] = "Statement " . ($index + 1) . ": " . $e->getMessage();
            echo "<div class='error'>✗ Error in statement " . ($index + 1) . ": " . htmlspecialchars($e->getMessage()) . "</div>";
        }
    }
    
    echo "<hr>";
    echo "<h2>📊 Installation Summary</h2>";
    echo "<p><strong>Successful statements:</strong> $successCount</p>";
    echo "<p><strong>Failed statements:</strong> $errorCount</p>";
    
    if ($errorCount > 0) {
        echo "<div class='error'><strong>Some errors occurred:</strong><br>";
        foreach ($errors as $error) {
            echo "• " . htmlspecialchars($error) . "<br>";
        }
        echo "</div>";
    }
    
    // Verify installation
    echo "<h2>🔍 Verifying Installation...</h2>";
    $stmt = $pdo->query("SHOW TABLES");
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (count($tables) > 0) {
        echo "<div class='success'><strong>✓ Installation Successful!</strong></div>";
        echo "<p>Created " . count($tables) . " tables:</p>";
        echo "<ul>";
        foreach ($tables as $table) {
            echo "<li>" . htmlspecialchars($table) . "</li>";
        }
        echo "</ul>";
        
        echo "<hr>";
        echo "<h2>✅ Next Steps</h2>";
        echo "<p>The database has been installed successfully. You can now:</p>";
        echo "<ol>";
        echo "<li>Create an admin user (if needed)</li>";
        echo "<li><a href='index.php' class='btn'>Go to Home Page</a></li>";
        echo "<li><a href='login.php' class='btn'>Login</a></li>";
        echo "</ol>";
    } else {
        echo "<div class='error'>✗ No tables were created. Please check the errors above.</div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'>";
    echo "<strong>✗ Installation Failed!</strong><br>";
    echo "<strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "<br>";
    echo "<strong>File:</strong> " . $e->getFile() . "<br>";
    echo "<strong>Line:</strong> " . $e->getLine() . "<br>";
    echo "</div>";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
}

echo "</body></html>";
