<?php
// Diagnostic tool to find redirect issues
ini_set('display_errors', 1);
error_reporting(E_ALL);

echo "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Redirect Diagnostic</title>
    <style>
        body { font-family: monospace; padding: 20px; background: #f5f5f5; }
        .info { background: white; padding: 15px; margin: 10px 0; border-radius: 5px; border-left: 4px solid #007bff; }
        .success { border-left-color: #28a745; }
        .error { border-left-color: #dc3545; }
        pre { background: #f8f9fa; padding: 10px; overflow-x: auto; }
        h2 { color: #333; margin-top: 30px; }
    </style>
</head>
<body>";

echo "<h1>🔍 Redirect Diagnostic Tool</h1>";

// 1. Check current URL
echo "<h2>1. Current Request Info</h2>";
echo "<div class='info'>";
echo "<strong>Request URI:</strong> " . ($_SERVER['REQUEST_URI'] ?? 'N/A') . "<br>";
echo "<strong>Script Name:</strong> " . ($_SERVER['SCRIPT_NAME'] ?? 'N/A') . "<br>";
echo "<strong>HTTP Host:</strong> " . ($_SERVER['HTTP_HOST'] ?? 'N/A') . "<br>";
echo "<strong>Server Name:</strong> " . ($_SERVER['SERVER_NAME'] ?? 'N/A') . "<br>";
echo "<strong>Document Root:</strong> " . ($_SERVER['DOCUMENT_ROOT'] ?? 'N/A') . "<br>";
echo "<strong>Request Method:</strong> " . ($_SERVER['REQUEST_METHOD'] ?? 'N/A') . "<br>";
echo "</div>";

// 2. Check session
session_start();
echo "<h2>2. Session Status</h2>";
echo "<div class='info'>";
echo "<strong>Session ID:</strong> " . session_id() . "<br>";
echo "<strong>Session Status:</strong> " . (session_status() === PHP_SESSION_ACTIVE ? 'Active' : 'Inactive') . "<br>";
echo "<strong>Admin Logged In:</strong> " . (isset($_SESSION['admin_id']) ? 'YES (ID: ' . $_SESSION['admin_id'] . ')' : 'NO') . "<br>";
echo "<strong>Coordinator Logged In:</strong> " . (isset($_SESSION['coordinator_id']) ? 'YES (ID: ' . $_SESSION['coordinator_id'] . ')' : 'NO') . "<br>";
echo "<strong>All Session Data:</strong><pre>" . print_r($_SESSION, true) . "</pre>";
echo "</div>";

// 3. Check installation status
echo "<h2>3. Installation Status</h2>";
try {
    $container = require __DIR__ . '/../bootstrap.php';
    $database = $container['database'];
    
    require_once __DIR__ . '/../src/Installer.php';
    $installer = new \Parking\Installer(dirname(__DIR__));
    $isInstalled = $installer->isInstalled($database);
    
    echo "<div class='info success'>";
    echo "<strong>Bootstrap:</strong> ✓ Loaded<br>";
    echo "<strong>Database Connection:</strong> ✓ Connected<br>";
    echo "<strong>Installation Status:</strong> " . ($isInstalled ? '✓ INSTALLED' : '✗ NOT INSTALLED') . "<br>";
    echo "</div>";
    
    // Check tables
    $pdo = $database->connection();
    $stmt = $pdo->query("SHOW TABLES");
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    echo "<div class='info'>";
    echo "<strong>Tables in database:</strong> " . count($tables) . "<br>";
    echo "<ul>";
    foreach ($tables as $table) {
        echo "<li>" . htmlspecialchars($table) . "</li>";
    }
    echo "</ul>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div class='info error'>";
    echo "<strong>Error:</strong> " . htmlspecialchars($e->getMessage());
    echo "</div>";
}

// 4. Check .htaccess
echo "<h2>4. Apache Configuration</h2>";
echo "<div class='info'>";
$htaccessFiles = [
    dirname(__DIR__) . '/.htaccess',
    __DIR__ . '/.htaccess',
];

foreach ($htaccessFiles as $file) {
    if (file_exists($file)) {
        echo "<strong>Found .htaccess at:</strong> $file<br>";
        echo "<pre>" . htmlspecialchars(file_get_contents($file)) . "</pre>";
    }
}

if (!file_exists($htaccessFiles[0]) && !file_exists($htaccessFiles[1])) {
    echo "<strong>.htaccess:</strong> Not found<br>";
}
echo "</div>";

// 5. Test page access
echo "<h2>5. Test Page Access</h2>";
echo "<div class='info'>";
$pages = [
    'index.php' => 'Main Page (Vehicle Registration)',
    'login.php' => 'Admin Login',
    'coordinator-login.php' => 'Coordinator Login',
    'dashboard.php' => 'Dashboard (requires login)',
    'admin-dashboard.php' => 'Admin Dashboard (requires admin login)',
];

echo "<ul>";
foreach ($pages as $page => $description) {
    $url = "http://" . $_SERVER['HTTP_HOST'] . "/" . $page;
    echo "<li><a href='$url' target='_blank'>$page</a> - $description</li>";
}
echo "</ul>";
echo "</div>";

// 6. Clear session option
echo "<h2>6. Session Management</h2>";
echo "<div class='info'>";
if (isset($_GET['clear_session'])) {
    session_destroy();
    echo "<strong style='color: green;'>✓ Session cleared!</strong><br>";
    echo "<a href='diagnostic.php'>Reload</a><br>";
} else {
    echo "<a href='?clear_session=1' style='padding: 10px 20px; background: #dc3545; color: white; text-decoration: none; border-radius: 5px; display: inline-block;'>Clear Session</a><br>";
}
echo "</div>";

// 7. Recommendations
echo "<h2>7. Troubleshooting Steps</h2>";
echo "<div class='info'>";
echo "<ol>";
echo "<li>If you see 'Admin Logged In: YES' above but want to access the main page, click 'Clear Session'</li>";
echo "<li>Check if there are any .htaccess rules redirecting all traffic</li>";
echo "<li>Try accessing pages directly using the links in section 5</li>";
echo "<li>Check your Apache virtual host configuration for redirects</li>";
echo "</ol>";
echo "</div>";

echo "</body></html>";
