<?php
require_once __DIR__ . '/session_init.php';

use Parking\Auth;
use Parking\Installer;
use Parking\EventRepository;
use Parking\CoordinatorRepository;

$container = require __DIR__ . '/../bootstrap.php';
$database = $container['database'];

$installer = new Installer(dirname(__DIR__));

if (!$installer->isInstalled($database)) {
    header('Location: install.php');
    exit;
}

// Check if user is logged in (either as admin or coordinator)
$isAdmin = false;
$isCoordinator = false;
$userFullName = '';
$coordinatorEventId = null;

if (isset($_SESSION['admin_id'])) {
    // Admin logged in
    $auth = new Auth($database);
    $auth->requireAuth();
    $isAdmin = true;
    $userFullName = $auth->getAdminFullName();
} elseif (isset($_SESSION['coordinator_id'])) {
    // Coordinator logged in
    $isCoordinator = true;
    $coordinatorRepo = new CoordinatorRepository($database);
    $coordinator = $coordinatorRepo->getCoordinatorById($_SESSION['coordinator_id']);
    
    if (!$coordinator) {
        session_destroy();
        header('Location: coordinator-login.php');
        exit;
    }
    
    $userFullName = $coordinator['full_name'];
    $coordinatorEventId = $coordinator['event_id'];
} else {
    // Not logged in at all
    header('Location: login.php');
    exit;
}

$eventRepo = new EventRepository($database);

// Get event based on user type
if ($coordinatorEventId) {
    // Coordinator: show their specific event
    $activeEvent = $eventRepo->getEventById($coordinatorEventId);
} else {
    // Admin: show active event
    $activeEvent = $eventRepo->getActiveEvent();
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <title>لوحة متابعة السيارات - نظام الباركنج</title>
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/dashboard.css">
    <link rel="stylesheet" href="assets/admin-dashboard.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
</head>
<body class="rtl sidebar-fixed">
    <?php 
    $currentPage = 'dashboard';
    include __DIR__ . '/includes/sidebar.php'; 
    ?>

    <main class="main-content dashboard">
        <!-- Top Bar -->
        <div class="topbar">
            <div>
                <button class="btn" id="sidebarToggle" aria-label="القائمة" style="display: none;">
                    <span class="material-icons">menu</span>
                </button>
                <h1 class="page-title">لوحة متابعة السيارات</h1>
                <p class="page-subtitle">تحديث حي لحالات السيارات المسجلة - الإشعارات الصوتية مفعلة تلقائياً 🔊</p>
            </div>
        </div>

        <!-- Event Banner -->
        <?php if ($activeEvent): ?>
            <div style="background: linear-gradient(135deg, rgba(108, 99, 255, 0.08) 0%, rgba(0, 217, 166, 0.06) 100%); border: 1px solid rgba(108, 99, 255, 0.15); border-radius: 16px; padding: 16px 24px; margin-bottom: 24px; display: flex; align-items: center; gap: 20px;">
                <?php if (!empty($activeEvent['logo'])): ?>
                    <div style="flex-shrink: 0;">
                        <img src="<?php echo htmlspecialchars($activeEvent['logo']); ?>" alt="Event Logo" style="height: 60px; width: auto; max-width: 120px; object-fit: contain; filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));" />
                    </div>
                <?php endif; ?>
                
                <div style="flex: 1; min-width: 0;">
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 6px;">
                        <span class="material-icons" style="font-size: 20px; color: var(--primary);">event</span>
                        <h3 style="font-size: 20px; font-weight: 700; color: var(--primary); margin: 0;">
                            <?php echo htmlspecialchars($activeEvent['name']); ?>
                        </h3>
                    </div>
                    
                    <div style="display: flex; flex-wrap: wrap; gap: 16px; font-size: 13px; color: var(--gray);">
                        <?php if ($activeEvent['event_date']): ?>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--primary);">calendar_today</span>
                                <span><?php echo htmlspecialchars($activeEvent['event_date']); ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($activeEvent['event_location']): ?>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--success);">location_on</span>
                                <span><?php echo htmlspecialchars($activeEvent['event_location']); ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($activeEvent['contact_phone']): ?>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span class="material-icons" style="font-size: 16px; color: var(--warning);">phone</span>
                                <span><?php echo htmlspecialchars($activeEvent['contact_phone']); ?></span>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Stats Cards -->
        <div class="stats-grid" style="grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); margin-bottom: 30px;">
            <div class="stat-card">
                <div class="stat-title">إجمالي السيارات النشطة</div>
                <div class="stat-value" id="stat-total">0</div>
            </div>
            <div class="stat-card">
                <div class="stat-title">طلبات قيد التجهيز</div>
                <div class="stat-value" id="stat-requested">0</div>
            </div>
            <div class="stat-card">
                <div class="stat-title">آخر تحديث</div>
                <div class="stat-value" id="stat-updated" style="font-size: 18px;">—</div>
            </div>
        </div>

        <div id="notifications" class="notifications" aria-live="polite"></div>

        <section class="table-wrapper">
            <table id="tickets-table">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>اسم العميل</th>
                        <th>رقم الهاتف</th>
                        <th>رقم اللوحة</th>
                        <th>اسم السائق</th>
                        <th>صور</th>
                        <th>الحالة</th>
                        <th>الإجراءات</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td colspan="8" class="empty">لا توجد بيانات بعد.</td>
                    </tr>
                </tbody>
            </table>
        </section>
    </main>
    
    <div class="sidebar-backdrop" id="sidebarBackdrop"></div>
    
    <script src="assets/sidebar.js"></script>
    <script src="assets/dashboard.js"></script>
</body>
</html>
