<?php
require_once __DIR__ . '/session_init.php';

use Parking\Database;
use Parking\CoordinatorRepository;
use Parking\EventRepository;
use Parking\ParkingRepository;

// Check if coordinator is logged in
if (!isset($_SESSION['coordinator_id'])) {
    header('Location: coordinator-login.php');
    exit;
}

$container = require __DIR__ . '/../bootstrap.php';
$database = $container['database'];

$coordinatorRepo = new CoordinatorRepository($database);
$eventRepo = new EventRepository($database);
$parkingRepo = new ParkingRepository($database);

$coordinatorId = $_SESSION['coordinator_id'];
$coordinator = $coordinatorRepo->getCoordinatorById($coordinatorId);

if (!$coordinator) {
    session_destroy();
    header('Location: coordinator-login.php');
    exit;
}

// Get coordinator statistics (scoped to this coordinator)
$stats = $coordinatorRepo->getCoordinatorStats($coordinatorId);

// Get statistics scoped to coordinator's event
$pdo = $database->connection();
$eventId = $coordinator['event_id'] ?? null;

if ($eventId) {
    $totalStmt = $pdo->prepare("SELECT COUNT(*) as total FROM tickets WHERE event_id = :eid AND status != 'returned'");
    $totalStmt->execute([':eid' => $eventId]);
    $total = $totalStmt->fetch()['total'] ?? 0;

    $parkedStmt = $pdo->prepare("SELECT COUNT(*) as total FROM tickets WHERE event_id = :eid AND status = 'parked'");
    $parkedStmt->execute([':eid' => $eventId]);
    $parked = $parkedStmt->fetch()['total'] ?? 0;

    $requestedStmt = $pdo->prepare("SELECT COUNT(*) as total FROM tickets WHERE event_id = :eid AND status = 'requested'");
    $requestedStmt->execute([':eid' => $eventId]);
    $requested = $requestedStmt->fetch()['total'] ?? 0;
} else {
    $total = $parked = $requested = 0;
}

// Get recent tickets by this coordinator (using coordinator_id, not valet_name)
$recentStmt = $pdo->prepare("SELECT * FROM tickets WHERE coordinator_id = :cid ORDER BY created_at DESC LIMIT 10");
$recentStmt->execute([':cid' => $coordinatorId]);
$recentTickets = $recentStmt->fetchAll();

// Get active event for sidebar
$activeEvent = null;
if ($eventId) {
    $activeEvent = $eventRepo->getEventById($eventId);
}

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>لوحة تحكم المنسق - <?php echo htmlspecialchars($coordinator['full_name']); ?></title>
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/admin-dashboard.css">
</head>
<body class="sidebar-fixed">
    <?php
    $currentPage = 'coordinator-dashboard';
    $isAdmin = false;
    $userFullName = $coordinator['full_name'];
    include __DIR__ . '/includes/sidebar.php';
    ?>

    <!-- Main Content -->
    <main class="main-content">
        <div class="topbar">
            <div>
                <button class="btn" id="sidebarToggle" aria-label="القائمة" style="display:none;">
                    <span class="material-icons">menu</span>
                </button>
                <h1 class="page-title">مرحباً، <?php echo htmlspecialchars($coordinator['full_name']); ?>!</h1>
                <p class="page-subtitle">إليك ملخص أعمالك اليوم</p>
            </div>
            <div class="topbar-actions">
                <button class="btn btn-primary" onclick="window.location.href='register-vehicle.php'">
                    <span class="material-icons" style="font-size: 18px;">add</span>
                    تسجيل سيارة جديدة
                </button>
            </div>
        </div>

        <!-- Stats Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-icon primary">
                        <span class="material-icons">assignment</span>
                    </div>
                </div>
                <div class="stat-title">إجمالي السيارات التي سجلتها</div>
                <div class="stat-value"><?php echo $stats['total'] ?? 0; ?></div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-icon success">
                        <span class="material-icons">local_parking</span>
                    </div>
                </div>
                <div class="stat-title">السيارات الموقفة</div>
                <div class="stat-value"><?php echo $stats['parked'] ?? 0; ?></div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-icon warning">
                        <span class="material-icons">schedule</span>
                    </div>
                </div>
                <div class="stat-title">السيارات قيد التجهيز</div>
                <div class="stat-value"><?php echo $stats['requested'] ?? 0; ?></div>
            </div>

            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-icon info">
                        <span class="material-icons">check_circle</span>
                    </div>
                </div>
                <div class="stat-title">السيارات المُسلمة</div>
                <div class="stat-value"><?php echo $stats['returned'] ?? 0; ?></div>
            </div>
        </div>

        <!-- Event Statistics -->
        <h2 style="color: var(--light); font-size: 24px; margin: 40px 0 20px; font-weight: 700;">إحصائيات الفعالية</h2>
        
        <div class="stats-grid" style="grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));">
            <div class="stat-card">
                <div class="stat-title">إجمالي السيارات النشطة</div>
                <div class="stat-value"><?php echo $total; ?></div>
            </div>

            <div class="stat-card">
                <div class="stat-title">السيارات الموقفة حالياً</div>
                <div class="stat-value"><?php echo $parked; ?></div>
            </div>

            <div class="stat-card">
                <div class="stat-title">السيارات قيد التجهيز</div>
                <div class="stat-value"><?php echo $requested; ?></div>
            </div>
        </div>

        <!-- Recent Tickets -->
        <div class="card" style="margin-top: 40px;">
            <div class="card-header">
                <h2 class="card-title">السيارات التي سجلتها مؤخراً</h2>
                <div class="card-actions">
                    <button class="btn btn-primary" onclick="window.location.href='dashboard.php'">
                        عرض الكل
                    </button>
                </div>
            </div>
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>العميل</th>
                            <th>رقم اللوحة</th>
                            <th>الحالة</th>
                            <th>الوقت</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($recentTickets)): ?>
                            <tr>
                                <td colspan="5" style="text-align: center; padding: 40px; color: var(--gray);">
                                    <span class="material-icons" style="font-size: 48px; opacity: 0.3;">inbox</span>
                                    <br>لم تقم بتسجيل أي سيارات بعد
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($recentTickets as $ticket): ?>
                                <tr>
                                    <td><strong>#<?php echo $ticket['id']; ?></strong></td>
                                    <td><?php echo htmlspecialchars($ticket['customer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td><strong><?php echo htmlspecialchars($ticket['license_plate'], ENT_QUOTES, 'UTF-8'); ?></strong></td>
                                    <td>
                                        <?php
                                        $statusLabels = [
                                            'parked' => ['موقف', 'info'],
                                            'requested' => ['قيد التجهيز', 'warning'],
                                            'returned' => ['تم التسليم', 'success']
                                        ];
                                        $status = $statusLabels[$ticket['status']] ?? ['غير معروف', 'danger'];
                                        ?>
                                        <span class="badge <?php echo $status[1]; ?>">
                                            <?php echo $status[0]; ?>
                                        </span>
                                    </td>
                                    <td style="font-size: 12px; color: var(--gray);">
                                        <?php 
                                        $time = strtotime($ticket['created_at']);
                                        echo date('H:i', $time);
                                        ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <div class="sidebar-backdrop" id="sidebarBackdrop"></div>
    
    <script src="assets/sidebar.js"></script>
    <script>
        // Auto refresh every 30 seconds
        setInterval(() => {
            location.reload();
        }, 30000);
    </script>
</body>
</html>
