<?php
require_once __DIR__ . '/session_init.php';

use Parking\Auth;
use Parking\Database;
use Parking\ClientRepository;

$container = require __DIR__ . '/../bootstrap.php';
$database = $container['database'];

$auth = new Auth($database);
$auth->requireAuth();

$isAdmin = true;
$userFullName = $auth->getAdminFullName();

$clientRepo = new ClientRepository($database);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        // Handle logo upload
        $logo = null;
        if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = __DIR__ . '/assets/clients/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $ext = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
            $filename = 'client_' . time() . '.' . $ext;
            $uploadPath = $uploadDir . $filename;
            
            if (move_uploaded_file($_FILES['logo']['tmp_name'], $uploadPath)) {
                $logo = 'assets/clients/' . $filename;
            }
        }
        
        $clientRepo->createClient([
            'name' => $_POST['name'],
            'logo' => $logo,
            'primary_color' => $_POST['primary_color'],
            'secondary_color' => $_POST['secondary_color'],
            'contact_person' => $_POST['contact_person'],
            'contact_phone' => $_POST['contact_phone'],
            'contact_email' => $_POST['contact_email']
        ]);
        
        header('Location: clients.php?success=created');
        exit;
    }
    
    if ($action === 'update') {
        $id = $_POST['id'];
        $client = $clientRepo->getClientById($id);
        
        // Handle logo upload
        $logo = $client['logo'];
        if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = __DIR__ . '/assets/clients/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $ext = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
            $filename = 'client_' . time() . '.' . $ext;
            $uploadPath = $uploadDir . $filename;
            
            if (move_uploaded_file($_FILES['logo']['tmp_name'], $uploadPath)) {
                $logo = 'assets/clients/' . $filename;
            }
        }
        
        $clientRepo->updateClient($id, [
            'name' => $_POST['name'],
            'logo' => $logo,
            'primary_color' => $_POST['primary_color'],
            'secondary_color' => $_POST['secondary_color'],
            'contact_person' => $_POST['contact_person'],
            'contact_phone' => $_POST['contact_phone'],
            'contact_email' => $_POST['contact_email']
        ]);
        
        header('Location: clients.php?success=updated');
        exit;
    }
    
    if ($action === 'toggle') {
        $id = $_POST['id'];
        $clientRepo->toggleClientStatus($id);
        header('Location: clients.php?success=toggled');
        exit;
    }
    
    if ($action === 'delete') {
        $id = $_POST['id'];
        $clientRepo->deleteClient($id);
        header('Location: clients.php?success=deleted');
        exit;
    }
    
    if ($action === 'switch') {
        $id = $_POST['id'];
        $clientRepo->switchClient($id);
        header('Location: admin-dashboard.php');
        exit;
    }
}

$activeEvent = null;
$clients = $clientRepo->getAllClients();
$activeClient = $clientRepo->getActiveClient();

?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الشركات - نظام الباركنج</title>
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <link rel="stylesheet" href="assets/theme.css">
    <link rel="stylesheet" href="assets/admin-dashboard.css">
    <style>
        .clients-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 24px;
            margin-top: 24px;
        }
        
        .client-card {
            background: var(--dark-card);
            border-radius: 16px;
            padding: 24px;
            border: 2px solid rgba(255, 255, 255, 0.06);
            transition: all 0.3s ease;
            position: relative;
        }
        
        .client-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: 0 8px 24px rgba(99, 102, 241, 0.15);
        }
        
        .client-card.active {
            border-color: var(--success);
            background: linear-gradient(135deg, var(--dark-card) 0%, rgba(16, 185, 129, 0.05) 100%);
        }
        
        .client-header {
            display: flex;
            align-items: center;
            gap: 16px;
            margin-bottom: 20px;
        }
        
        .client-logo {
            width: 64px;
            height: 64px;
            border-radius: 12px;
            object-fit: contain;
            background: rgba(255, 255, 255, 0.05);
            padding: 8px;
        }
        
        .client-info h3 {
            margin: 0 0 4px 0;
            color: var(--light);
            font-size: 18px;
        }
        
        .client-event {
            color: var(--gray);
            font-size: 14px;
        }
        
        .client-details {
            margin: 16px 0;
            padding: 16px;
            background: rgba(255, 255, 255, 0.02);
            border-radius: 10px;
        }
        
        .client-detail {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 8px;
            font-size: 13px;
            color: var(--gray);
        }
        
        .client-detail:last-child {
            margin-bottom: 0;
        }
        
        .client-colors {
            display: flex;
            gap: 8px;
            margin: 12px 0;
        }
        
        .color-box {
            width: 40px;
            height: 40px;
            border-radius: 8px;
            border: 2px solid rgba(255, 255, 255, 0.1);
        }
        
        .client-actions {
            display: flex;
            gap: 8px;
            margin-top: 16px;
        }
        
        .badge-active {
            position: absolute;
            top: 16px;
            left: 16px;
            background: var(--success);
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(4px);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }
        
        .modal.show {
            display: flex;
        }
        
        .modal-content {
            background: var(--dark-card);
            border-radius: 16px;
            padding: 32px;
            max-width: 600px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 24px;
        }
        
        .modal-header h2 {
            margin: 0;
            color: var(--light);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: var(--light);
            font-weight: 600;
            font-size: 14px;
        }
        
        .form-group input,
        .form-group textarea {
            width: 100%;
            padding: 12px;
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            color: var(--light);
            font-family: 'Tajawal', sans-serif;
        }
        
        .form-group input[type="color"] {
            height: 50px;
            cursor: pointer;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 16px;
        }
    </style>
</head>
<body class="sidebar-fixed">
    <?php 
    $currentPage = 'clients';
    include __DIR__ . '/includes/sidebar.php'; 
    ?>

    <main class="main-content">
        <div class="topbar">
            <div>
                <h1 class="page-title">إدارة الشركات</h1>
                <p class="page-subtitle">إدارة عملاء الخدمة والإيفنتات</p>
            </div>
            <div class="topbar-actions">
                <button class="btn btn-primary" onclick="openModal()">
                    <span class="material-icons" style="font-size: 18px;">add</span>
                    إضافة شركة جديدة
                </button>
            </div>
        </div>

        <?php if (isset($_GET['success'])): ?>
            <div class="alert alert-success" style="margin-bottom: 24px;">
                <?php
                $messages = [
                    'created' => 'تم إضافة الشركة بنجاح',
                    'updated' => 'تم تحديث الشركة بنجاح',
                    'deleted' => 'تم حذف الشركة بنجاح',
                    'toggled' => 'تم تحديث حالة الشركة'
                ];
                echo $messages[$_GET['success']] ?? 'تمت العملية بنجاح';
                ?>
            </div>
        <?php endif; ?>

        <div class="clients-grid">
            <?php foreach ($clients as $client): ?>
                <div class="client-card <?php echo $client['id'] == $activeClient['id'] ? 'active' : ''; ?>">
                    <?php if ($client['id'] == $activeClient['id']): ?>
                        <span class="badge-active">
                            <span class="material-icons" style="font-size: 14px; vertical-align: middle;">check_circle</span>
                            نشط حالياً
                        </span>
                    <?php endif; ?>
                    
                    <div class="client-header">
                        <img src="<?php echo htmlspecialchars($client['logo'] ?? 'assets/logo.png'); ?>" alt="Logo" class="client-logo">
                        <div class="client-info">
                            <h3><?php echo htmlspecialchars($client['name']); ?></h3>
                        </div>
                    </div>
                    
                    <div class="client-details">
                        <?php if ($client['contact_person']): ?>
                                <div class="client-detail">
                                    <span class="material-icons" style="font-size: 16px;">person</span>
                                    <?php echo htmlspecialchars($client['contact_person']); ?>
                                </div>
                            <?php endif; ?>
                            <?php if ($client['contact_phone']): ?>
                                <div class="client-detail">
                                    <span class="material-icons" style="font-size: 16px;">phone</span>
                                    <?php echo htmlspecialchars($client['contact_phone']); ?>
                                </div>
                            <?php endif; ?>
                    </div>
                    
                    <div class="client-colors">
                        <div class="color-box" style="background: <?php echo htmlspecialchars($client['primary_color']); ?>"></div>
                        <div class="color-box" style="background: <?php echo htmlspecialchars($client['secondary_color']); ?>"></div>
                    </div>
                    
                    <div class="client-actions">
                        <?php if ($client['id'] != $activeClient['id']): ?>
                            <form method="POST" style="flex: 1;">
                                <input type="hidden" name="action" value="switch">
                                <input type="hidden" name="id" value="<?php echo $client['id']; ?>">
                                <button type="submit" class="btn btn-primary" style="width: 100%;">تفعيل</button>
                            </form>
                        <?php endif; ?>
                        <button class="btn" onclick="editClient(<?php echo htmlspecialchars(json_encode($client)); ?>)">
                            <span class="material-icons" style="font-size: 16px;">edit</span>
                        </button>
                        <?php if ($client['id'] != 1): ?>
                            <form method="POST" onsubmit="return confirm('هل أنت متأكد من الحذف؟')">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="id" value="<?php echo $client['id']; ?>">
                                <button type="submit" class="btn" style="color: var(--danger);">
                                    <span class="material-icons" style="font-size: 16px;">delete</span>
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </main>

    <!-- Add/Edit Modal -->
    <div class="modal" id="clientModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitle">إضافة شركة جديدة</h2>
                <button class="btn" onclick="closeModal()">
                    <span class="material-icons">close</span>
                </button>
            </div>
            
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="id" id="clientId">
                
                <div class="form-group">
                    <label>اسم الشركة *</label>
                    <input type="text" name="name" id="clientName" required>
                </div>
                
                <div class="form-group">
                    <label>الشعار</label>
                    <input type="file" name="logo" accept="image/*">
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>اللون الأساسي</label>
                        <input type="color" name="primary_color" id="primaryColor" value="#6366f1">
                    </div>
                    <div class="form-group">
                        <label>اللون الثانوي</label>
                        <input type="color" name="secondary_color" id="secondaryColor" value="#10b981">
                    </div>
                </div>
                
                <div class="form-group">
                    <label>الشخص المسؤول</label>
                    <input type="text" name="contact_person" id="contactPerson">
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>رقم الهاتف</label>
                        <input type="text" name="contact_phone" id="contactPhone">
                    </div>
                    <div class="form-group">
                        <label>البريد الإلكتروني</label>
                        <input type="email" name="contact_email" id="contactEmail">
                    </div>
                </div>
                
                <div style="display: flex; gap: 12px; margin-top: 24px;">
                    <button type="submit" class="btn btn-primary" style="flex: 1;">حفظ</button>
                    <button type="button" class="btn" onclick="closeModal()">إلغاء</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function openModal() {
            document.getElementById('clientModal').classList.add('show');
            document.getElementById('modalTitle').textContent = 'إضافة شركة جديدة';
            document.getElementById('formAction').value = 'create';
            document.querySelector('form').reset();
        }
        
        function closeModal() {
            document.getElementById('clientModal').classList.remove('show');
        }
        
        function editClient(client) {
            document.getElementById('clientModal').classList.add('show');
            document.getElementById('modalTitle').textContent = 'تعديل الشركة';
            document.getElementById('formAction').value = 'update';
            document.getElementById('clientId').value = client.id;
            document.getElementById('clientName').value = client.name;
            document.getElementById('primaryColor').value = client.primary_color;
            document.getElementById('secondaryColor').value = client.secondary_color;
            document.getElementById('contactPerson').value = client.contact_person || '';
            document.getElementById('contactPhone').value = client.contact_phone || '';
            document.getElementById('contactEmail').value = client.contact_email || '';
        }
        
        // Close modal on backdrop click
        document.getElementById('clientModal').addEventListener('click', function(e) {
            if (e.target === this) closeModal();
        });
    </script>
    <script src="assets/sidebar.js"></script>
</body>
</html>
