const tableBody = document.querySelector('#tickets-table tbody');
const notificationsContainer = document.getElementById('notifications');
const statTotal = document.getElementById('stat-total');
const statRequested = document.getElementById('stat-requested');
const statUpdated = document.getElementById('stat-updated');
const enableSoundBtn = document.getElementById('enableSoundBtn');

let lastTickets = new Map();
let eventSource;
let reconnectTimeout;
let reconnectAttempts = 0;
let audioContext;
let soundEnabled = false;
let isReloading = false; // Prevent multiple reloads

tableBody.addEventListener('click', handleActionClick);
window.addEventListener('beforeunload', () => {
    if (eventSource) {
        eventSource.close();
    }
});

// Sound is enabled on first user interaction (not auto-play)

// Also enable on first user interaction (fallback)
let autoSoundEnabled = false;
const autoEnableSound = () => {
    if (!autoSoundEnabled && !soundEnabled) {
        autoSoundEnabled = true;
        enableSound();
    }
};

// Try to enable sound on any user interaction (fallback)
document.addEventListener('click', autoEnableSound, { once: true });
document.addEventListener('keydown', autoEnableSound, { once: true });
document.addEventListener('touchstart', autoEnableSound, { once: true });

initializeDashboard();

async function initializeDashboard() {
    await fetchTickets();
    startPolling(); // Changed from startStream() to polling every 15 seconds
}

async function fetchTickets() {
    try {
        const response = await fetch('api/tickets.php?scope=active', { cache: 'no-store' });
        if (!response.ok) {
            throw new Error('فشل تحميل البيانات (HTTP ' + response.status + ')');
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const text = await response.text();
            console.error('Response is not JSON:', text.substring(0, 200));
            throw new Error('استجابة غير صحيحة من الخادم');
        }
        
        const data = await response.json();
        const tickets = data.tickets ?? [];
        const timestamp = Date.now();
        
        // Initialize lastTickets on first load if empty
        if (lastTickets.size === 0) {
            lastTickets = new Map(tickets.map((ticket) => [Number(ticket.id), ticket]));
            updateDashboard(tickets, timestamp);
        } else {
            // Use handleLiveUpdate for subsequent updates to trigger sound notifications
            handleLiveUpdate(tickets, timestamp);
        }
    } catch (error) {
        console.error('Failed to load tickets', error);
        // Silent error - don't show toast
    }
}

// Polling function - refresh every 15 seconds
let pollingInterval = null;
function startPolling() {
    // Clear any existing interval
    if (pollingInterval) {
        clearInterval(pollingInterval);
    }
    
    // Fetch tickets every 15 seconds
    pollingInterval = setInterval(() => {
        fetchTickets();
    }, 15000); // 15 seconds
    
    console.log('✅ التحديث التلقائي كل 15 ثانية مفعّل');
}

// Stop polling when page is hidden/closed
window.addEventListener('beforeunload', () => {
    if (pollingInterval) {
        clearInterval(pollingInterval);
    }
    if (eventSource) {
        eventSource.close();
    }
});

// Old SSE functions - no longer used but kept for compatibility
function startStream() {
    // Disabled - using polling instead
}

function scheduleReconnect() {
    // Disabled - using polling instead
}

function handleLiveUpdate(tickets, timestamp) {
    console.log('📊 handleLiveUpdate called with', tickets.length, 'tickets');
    
    const previous = lastTickets;
    const nextMap = new Map();

    tickets.forEach((ticket) => {
        const id = Number(ticket.id);
        nextMap.set(id, ticket);

        const previousTicket = previous.get(id);

        if (!previousTicket && ticket.status === 'requested') {
            console.log('🆕 تذكرة جديدة بحالة "طلب"! ID:', id, 'العميل:', ticket.customer_name);
            announceRequest(ticket);
        } else if (previousTicket && previousTicket.status !== ticket.status) {
            console.log('🔄 تغيير حالة التذكرة ID:', id, 'من', previousTicket.status, 'إلى', ticket.status);
            handleStatusChange(previousTicket, ticket);
        }
    });

    lastTickets = nextMap;
    updateDashboard(tickets, timestamp);
}

function updateDashboard(tickets, timestamp) {
    renderTickets(tickets);
    updateStats(tickets, timestamp);
}

function renderTickets(tickets) {
    if (!tickets.length) {
        if (tableBody.children.length !== 1 || !tableBody.querySelector('.empty')) {
            tableBody.innerHTML = '<tr><td colspan="8" class="empty">لا توجد بيانات بعد.</td></tr>';
        }
        return;
    }

    // Get existing rows mapped by ticket ID
    const existingRows = new Map();
    Array.from(tableBody.querySelectorAll('tr[data-ticket-id]')).forEach(row => {
        existingRows.set(row.dataset.ticketId, row);
    });

    // Track which tickets we've processed
    const processedIds = new Set();

    tickets.forEach((ticket, index) => {
        processedIds.add(ticket.id.toString());
        const existingRow = existingRows.get(ticket.id.toString());
        
        if (existingRow) {
            // Update existing row only if data changed
            updateRowIfNeeded(existingRow, ticket, index);
        } else {
            // Create new row
            const row = createTicketRow(ticket, index);
            tableBody.appendChild(row);
        }
    });

    // Remove rows for tickets that no longer exist (smoothly)
    existingRows.forEach((row, ticketId) => {
        if (!processedIds.has(ticketId)) {
            row.style.transition = 'opacity 0.3s ease';
            row.style.opacity = '0';
            setTimeout(() => {
                if (row.parentNode) {
                    row.remove();
                }
            }, 300);
        }
    });

    // SKIP re-ordering completely - just update row numbers
    // This prevents any visual flickering from DOM reordering
    // Row numbers are already updated in updateRowIfNeeded()
}

function createTicketRow(ticket, index) {
    const row = document.createElement('tr');
    row.dataset.ticketId = ticket.id;
    
    // Store ticket data for comparison in future updates
    row.dataset.ticketData = JSON.stringify({
        status: ticket.status,
        updated_at: ticket.updated_at || ticket.created_at,
        customer_name: ticket.customer_name,
        phone_number: ticket.phone_number,
        license_plate: ticket.license_plate,
        valet_name: ticket.valet_name
    });
    
    if (ticket.status === 'requested') {
        row.classList.add('ticket-requested');
    }
    
    const statusClass = ['parked', 'requested', 'returned'].includes(ticket.status) ? ticket.status : 'parked';
    row.innerHTML = `
        <td>${index + 1}</td>
        <td>${sanitize(ticket.customer_name)}</td>
        <td>${sanitize(ticket.phone_number)}</td>
        <td>${sanitize(ticket.license_plate)}</td>
        <td>${sanitize(ticket.valet_name)}</td>
        <td>${renderPhotos(ticket.photos)}</td>
        <td><span class="status-pill ${statusClass}">${formatStatus(ticket.status)}</span></td>
        <td class="actions">${renderActions(ticket)}</td>
    `;
    
    // No fade animation - instant display to prevent flicker
    return row;
}

function updateRowIfNeeded(row, ticket, index) {
    // Store previous ticket data to compare
    const prevDataStr = row.dataset.ticketData || '{}';
    const prevData = JSON.parse(prevDataStr);
    
    const currentData = {
        status: ticket.status,
        updated_at: ticket.updated_at || ticket.created_at,
        customer_name: ticket.customer_name,
        phone_number: ticket.phone_number,
        license_plate: ticket.license_plate,
        valet_name: ticket.valet_name
    };
    
    // Calculate if anything changed
    const statusChanged = prevData.status !== ticket.status;
    const timeChanged = prevData.updated_at !== currentData.updated_at;
    const indexChanged = row.children[0].textContent !== (index + 1).toString();
    
    // If nothing changed, skip completely
    if (!statusChanged && !timeChanged && !indexChanged) {
        return;
    }
    
    // Store new data
    row.dataset.ticketData = JSON.stringify(currentData);
    
    // Update ONLY the row number if it changed
    if (indexChanged) {
        row.children[0].textContent = index + 1;
    }
    
    // Update ONLY the status-related elements if status changed
    if (statusChanged) {
        const statusClass = ['parked', 'requested', 'returned'].includes(ticket.status) ? ticket.status : 'parked';
        
        // Update row class for requested tickets
        if (ticket.status === 'requested') {
            row.classList.add('ticket-requested');
        } else {
            row.classList.remove('ticket-requested');
        }
        
        // Update status pill (now at index 6, since we removed time column)
        const statusCell = row.children[6];
        if (statusCell) {
            const currentPill = statusCell.querySelector('.status-pill');
            const newClass = `status-${statusClass}`;
            const newText = formatStatus(ticket.status);
            
            // Only update if actually different
            if (!currentPill || !currentPill.classList.contains(newClass)) {
                statusCell.innerHTML = `<span class="status-pill ${newClass}">${newText}</span>`;
            }
        }
        
        // Update actions cell (now at index 7, since we removed time column)
        const actionsCell = row.children[7];
        if (actionsCell) {
            const newActions = renderActions(ticket);
            const currentActions = actionsCell.innerHTML.trim();
            if (currentActions !== newActions.trim()) {
                actionsCell.innerHTML = newActions;
            }
        }
    }
    
    // NO timestamp column anymore - removed!
}

function renderActions(ticket) {
    if (ticket.status === 'requested') {
        return '<button class="btn" data-action="returned">تسليم السيارة</button>';
    }
    if (ticket.status === 'parked') {
        return '<button class="btn ghost" data-action="requested">بدء التجهيز</button>';
    }

    return '<span class="status-pill status-returned">مكتمل</span>';
}

function updateStats(tickets, timestamp) {
    const total = tickets.length;
    const requested = tickets.filter((ticket) => ticket.status === 'requested').length;

    statTotal.textContent = total;
    statRequested.textContent = requested;
    statUpdated.textContent = formatTimestamp(timestamp);
}

function handleStatusChange(previousTicket, currentTicket) {
    if (currentTicket.status === 'requested' && previousTicket.status !== 'requested') {
        announceRequest(currentTicket);
        return;
    }

    if (currentTicket.status === 'returned') {
        showToast(`🚗 تم تسليم السيارة للعميل ${sanitize(currentTicket.customer_name)}.`, 'success');
    }

    if (currentTicket.status === 'parked' && previousTicket.status !== 'parked') {
        showToast(`✅ تمت إعادة السيارة إلى الحظيرة (${sanitize(currentTicket.license_plate)}).`, 'success');
    }
}

function announceRequest(ticket) {
    console.log('🛎️ طلب جديد! العميل:', ticket.customer_name, 'السيارة:', ticket.license_plate);
    console.log('🔊 soundEnabled =', soundEnabled, '| سيتم تشغيل الصوت الآن...');
    playNotificationTone();
    showToast(`🛎️ العميل ${sanitize(ticket.customer_name)} طلب سيارته (${sanitize(ticket.license_plate)}).`, 'info');
}

async function updateTicketStatus(ticketId, status) {
    const response = await fetch('api/status.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ ticket_id: ticketId, status }),
    });

    if (!response.ok) {
        const data = await response.json().catch(() => ({}));
        const message = data.error ?? 'تعذر تحديث الحالة.';
        throw new Error(message);
    }

    return response.json();
}

async function handleActionClick(event) {
    const button = event.target.closest('button[data-action]');
    if (!button) {
        return;
    }

    const row = button.closest('tr');
    const ticketId = Number(row?.dataset.ticketId ?? 0);
    const action = button.dataset.action;

    if (!ticketId || !action) {
        return;
    }

    const originalText = button.textContent;
    button.disabled = true;
    button.textContent = 'جارٍ المعالجة...';

    try {
        await updateTicketStatus(ticketId, action);
        if (action === 'returned') {
            showToast('تم تسجيل تسليم السيارة بنجاح.', 'success');
        } else if (action === 'requested') {
            showToast('تم تحويل الحالة إلى "قيد التجهيز".', 'info');
        } else if (action === 'parked') {
            showToast('تم إعادة الحالة إلى "موقف".', 'success');
        }
    } catch (error) {
        showToast(error.message, 'error');
        button.disabled = false;
    } finally {
        button.textContent = originalText;
    }
}

function showToast(message, type = 'info') {
    if (!notificationsContainer) {
        return;
    }

    const toast = document.createElement('div');
    toast.className = `toast ${type}`;

    const content = document.createElement('div');
    content.innerHTML = `<strong>${message}</strong>`;

    const closeButton = document.createElement('button');
    closeButton.className = 'close-btn';
    closeButton.setAttribute('aria-label', 'إغلاق');
    closeButton.innerHTML = '×';
    closeButton.addEventListener('click', () => {
        notificationsContainer.removeChild(toast);
    });

    toast.appendChild(content);
    toast.appendChild(closeButton);
    notificationsContainer.appendChild(toast);

    setTimeout(() => {
        if (notificationsContainer.contains(toast)) {
            notificationsContainer.removeChild(toast);
        }
    }, 6000);
}

function enableSound() {
    try {
        console.log('🔊 محاولة تفعيل الصوت...');
        
        const AudioContextConstructor = window.AudioContext || window.webkitAudioContext;
        if (!AudioContextConstructor) {
            console.error('❌ المتصفح لا يدعم الإشعارات الصوتية');
            showToast('المتصفح لا يدعم الإشعارات الصوتية.', 'error');
            return;
        }

        audioContext = audioContext || new AudioContextConstructor();
        console.log('📢 AudioContext created. State:', audioContext.state);
        
        if (audioContext.state === 'suspended') {
            audioContext.resume().then(() => {
                console.log('▶️ AudioContext resumed successfully');
            });
        }

        soundEnabled = true;
        console.log('✅ الإشعارات الصوتية مفعّلة! soundEnabled =', soundEnabled);
        
        // Test sound to verify it works
        playNotificationTone();
        
    } catch (error) {
        console.error('❌ فشل تفعيل الصوت:', error);
    }
}

function playNotificationTone() {
    console.log('🔔 playNotificationTone called. soundEnabled =', soundEnabled);
    
    if (!soundEnabled) {
        console.warn('⚠️ الصوت معطّل! لن يتم التشغيل.');
        return;
    }
    
    console.log('🎵 تشغيل نغمة الإشعار...');
    
    try {
        const AudioContextConstructor = window.AudioContext || window.webkitAudioContext;
        if (!AudioContextConstructor) {
            console.error('❌ المتصفح لا يدعم AudioContext');
            return;
        }

        audioContext = audioContext || new AudioContextConstructor();
        console.log('🔊 AudioContext state:', audioContext.state);

        if (audioContext.state === 'suspended') {
            audioContext.resume().then(() => {
                console.log('▶️ AudioContext resumed for playback');
            });
        }

        // Play 3 beeps for attention
        const times = [0, 0.15, 0.3];
        times.forEach((startTime) => {
            const oscillator = audioContext.createOscillator();
            const gain = audioContext.createGain();

            oscillator.type = 'sine';
            oscillator.frequency.setValueAtTime(880, audioContext.currentTime + startTime);

            gain.gain.setValueAtTime(0, audioContext.currentTime + startTime);
            gain.gain.exponentialRampToValueAtTime(0.4, audioContext.currentTime + startTime + 0.02);
            gain.gain.exponentialRampToValueAtTime(0.0001, audioContext.currentTime + startTime + 0.12);

            oscillator.connect(gain).connect(audioContext.destination);
            oscillator.start(audioContext.currentTime + startTime);
            oscillator.stop(audioContext.currentTime + startTime + 0.12);
        });
        
        console.log('✅ نغمة الإشعار تم تشغيلها بنجاح!');
    } catch (error) {
        console.error('❌ تعذر تشغيل التنبيه الصوتي:', error);
    }
}

function renderPhotos(photosJson) {
    if (!photosJson) {
        return '<span style="color: var(--gray); font-size: 12px;">لا توجد</span>';
    }
    
    try {
        const photos = JSON.parse(photosJson);
        if (!photos || photos.length === 0) {
            return '<span style="color: var(--gray); font-size: 12px;">لا توجد</span>';
        }
        
        const photosHtml = photos.map((photo, index) => 
            `<img src="${sanitize(photo)}" 
                  alt="صورة ${index + 1}" 
                  style="width: 40px; height: 40px; object-fit: cover; border-radius: 6px; cursor: pointer; border: 2px solid rgba(99, 102, 241, 0.3);"
                  onclick="showPhotoModal('${sanitize(photo)}', ${JSON.stringify(photos).replace(/"/g, '&quot;')})"
            />`
        ).join(' ');
        
        return `<div style="display: flex; gap: 4px; flex-wrap: wrap;">${photosHtml}</div>`;
    } catch (e) {
        return '<span style="color: var(--danger); font-size: 12px;">خطأ</span>';
    }
}

function showPhotoModal(currentPhoto, allPhotosJson) {
    const allPhotos = JSON.parse(allPhotosJson.replace(/&quot;/g, '"'));
    const currentIndex = allPhotos.indexOf(currentPhoto);
    
    const modal = document.createElement('div');
    modal.style.cssText = `
        position: fixed; top: 0; left: 0; width: 100%; height: 100%; 
        background: rgba(0, 0, 0, 0.9); z-index: 9999; 
        display: flex; align-items: center; justify-content: center;
        backdrop-filter: blur(5px);
    `;
    
    modal.innerHTML = `
        <button onclick="this.parentElement.remove()" style="position: absolute; top: 20px; right: 20px; background: rgba(239, 68, 68, 0.9); color: white; border: none; border-radius: 50%; width: 48px; height: 48px; cursor: pointer; font-size: 24px; z-index: 10000;">
            ×
        </button>
        ${allPhotos.length > 1 ? `
            <button onclick="navigatePhoto(${currentIndex - 1}, ${JSON.stringify(allPhotos).replace(/"/g, '&quot;')}, this.parentElement)" style="position: absolute; left: 20px; background: rgba(99, 102, 241, 0.9); color: white; border: none; border-radius: 50%; width: 48px; height: 48px; cursor: pointer; font-size: 24px; ${currentIndex === 0 ? 'opacity: 0.3; pointer-events: none;' : ''}">
                ‹
            </button>
            <button onclick="navigatePhoto(${currentIndex + 1}, ${JSON.stringify(allPhotos).replace(/"/g, '&quot;')}, this.parentElement)" style="position: absolute; right: 20px; background: rgba(99, 102, 241, 0.9); color: white; border: none; border-radius: 50%; width: 48px; height: 48px; cursor: pointer; font-size: 24px; ${currentIndex === allPhotos.length - 1 ? 'opacity: 0.3; pointer-events: none;' : ''}">
                ›
            </button>
        ` : ''}
        <img id="modalPhoto" src="${sanitize(currentPhoto)}" style="max-width: 90%; max-height: 90%; border-radius: 12px; box-shadow: 0 20px 60px rgba(0,0,0,0.5);">
        <div style="position: absolute; bottom: 30px; color: white; font-size: 14px; background: rgba(0, 0, 0, 0.7); padding: 8px 16px; border-radius: 20px;">
            ${allPhotos.length > 1 ? `صورة ${currentIndex + 1} من ${allPhotos.length}` : 'صورة السيارة'}
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Close on click outside
    modal.addEventListener('click', (e) => {
        if (e.target === modal) {
            modal.remove();
        }
    });
    
    // Close on ESC key
    const handleEsc = (e) => {
        if (e.key === 'Escape') {
            modal.remove();
            document.removeEventListener('keydown', handleEsc);
        }
    };
    document.addEventListener('keydown', handleEsc);
}

function navigatePhoto(newIndex, allPhotosJson, modalElement) {
    const allPhotos = JSON.parse(allPhotosJson.replace(/&quot;/g, '"'));
    if (newIndex < 0 || newIndex >= allPhotos.length) return;
    
    modalElement.remove();
    showPhotoModal(allPhotos[newIndex], allPhotosJson);
}

function sanitize(value) {
    if (value == null) {
        return '';
    }
    return String(value).replace(/[&<>"']/g, (char) => ({
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#39;',
    }[char]));
}

function formatStatus(status) {
    switch (status) {
        case 'parked':
            return 'موقف';
        case 'requested':
            return 'قيد التجهيز';
        case 'returned':
            return 'مستلم';
        default:
            return status;
    }
}

function formatDate(value) {
    if (!value) {
        return '—';
    }
    const date = new Date(value.replace(' ', 'T') + 'Z');
    return date.toLocaleString('ar-EG', { hour12: false });
}

function formatTimestamp(value) {
    const date = new Date(value);
    if (Number.isNaN(date.getTime())) {
        return '—';
    }
    return date.toLocaleTimeString('ar-EG', { hour12: false });
}

function parseEventData(payload) {
    if (!payload || payload === 'undefined') {
        return {};
    }
    try {
        return JSON.parse(payload);
    } catch (error) {
        console.error('Unable to parse event payload:', error, 'Payload:', payload);
        return {};
    }
}
